;***************************************************************
;                                                              *
;  utils.s  Assembler general purpose utilities                *
;                                                              *
;                                                              *
;**************************************************************/
;       include "./xref.inc"    ; Cosmic external references

;         include "./ab_regs.inc" ; MC68HC908AB32 register defimitions
;         include "./macros.inc"
;         include "./local.inc"

;

;
#if DEVEL
;        section .devtext
#elseif
;        section .text
#endif
;
;     _sp_gets  Gets string  if 1st char a space it exits
;
;      export _sp_gets
_sp_gets:
      pshhx
      psha
      ldhx #_inbuf
      jsr _getch
      cmp #SPACE
      beq cancel_gets
      cmp #BS
      beq cancel_gets
      bra first_gets
;
;
;  Receive a String, No Interrupt Routine
;        Returns carry set if ^C, or ESC
;
;      export _gets
_gets:
      pshhx
      psha
      ldhx #_inbuf
inrlp:                          ; Fill receiver buffer with spaces
      clr ,x
      aix #1
      cphx #(_inbuf+RXBLEN)
      bne inrlp
      ldhx #_inbuf
wt_nxt_char:
      sta copctl                ; strobe Watch dog timer
      brclr SCRF,scs1,wt_nxt_char
      lda scdr                ; Read in byte
      jsr _putch
first_gets:
      cmp #CR
      bne notcr                 ; Is the entry over? (CR received)
                                ; If so, terminate entry with a null
      clr ,x                    ; Store null in buffer
      cphx #_inbuf
      bne rxdn_ok               ; If buffer not empty, exit with carry clear
      bra rxdn_err              ; If buffer empty, exit with carry set
notcr:
      cmp #$03                 ; ^C hit?
      bne not_cancel            ; If so store and buffer and terminate entry
cancel_gets:
      sta _inbuf
      clra
      sta _inbuf+1
rxdn_err:
      sec
      bra rxdn
not_cancel:
      cmp #$08
      bne notbs                 ; Is it a back space? (BS received)
      cphx #_inbuf
      beq rx_cancel             ; Don't baskspace past beginning
      aix #-1
      bra wt_nxt_char
notbs:
      sta ,x                  ; Store char in buffer
      aix #1
      cphx #_inbuf+RXBLEN-1      ; Is buffer full
      bne wt_nxt_char
rx_cancel:
      clr ,x
rxdn_ok:
      clc
rxdn:
      pula
      pulhx
      rts
;
;     a-reg ahex2b(h:x-reg)  Converts 2 ascii hex (h:x-reg) encodes byte
;                                to one byte hex (a-reg)
;      export _ahex2b
_ahex2b:
      txa
      bsr _a2nibhex     ; Convert low byte to ascii
      psha
      tha
      bsr _a2nibhex     ; Convert high byte to ascii
      nsa               ; put high byte in high nibble
      add 1,sp          ; Combine low and high nibbles
      pulx
      rts
;
;     Convert ASCII Byte to hex nibbl.
;
;      export _a2nibhex
_a2nibhex:
      cmp #'9'       ; If Ascii A-F, you need to add 9 to
      bls ah1        ; convert to hex, ie, A(41) + 9 -> 4A -> 0A
      add #$09
ah1:
      and #$0F
      rts
;
;  (double byte) dblchar(a-reg) passed a byte in a register
;                returns ascii encoded chars in h-x registers
;
;      export _dblchar
_dblchar:
      psha
      bsr _nib_to_asc
      tax
      pula
dblow:
      nsa
      bsr _nib_to_asc
      tah
      rts
;
;     Convert low nibble to ASCII encoded Hex
;
;      export _nib_to_asc
_nib_to_asc:
      and #$0F
      ora #$30
      cmp #$39
      ble dbend
      add #$07
dbend:
      rts
;
;     _comma_sp Output ,sp
;
;      export _comma_sp
_comma_sp:
      psha
      bsr _pr_comma
      lda #'s'
      bsr _putch
      lda #'p'
      bra out_byte
;
;     _pr_letter_x Output a pound sign (#)
;
;      export _pr_letter_x
_pr_letter_x:
      psha
      lda #'x'
      bra out_byte
;
;     _pr_comma Output a pound sign (#)
;
;      export _pr_pound
_pr_pound:
      psha
      lda #'#'
      bra out_byte

;
;     _pr_comma Output a comma
;
;      export _pr_comma
_pr_comma:
      psha
      lda #','
      bra out_byte
;
;     _crlf(unsigned char c) Output a cr & lf
;
;      export _crlf
_crlf:
      psha
      lda #CR
      bsr _putch
      lda #LF
      bra out_byte
;
;     _sp_byte  Prints a byte and then a space
;
;      export _byte_sp
_byte_sp:
      jsr _pbyte
;
;     _space(unsigned char c) Output a space
;
;      export _space
_space:
      psha
      lda #SPACE
out_byte:
      bsr _putch
      pula
      rts
;
;     _pr_nibble(unsigned char a-reg) Output Ascii encoded low nibble
;
;      export _pr_nibble
_pr_nibble:
      jsr _nib_to_asc
;
;      _putch(a-reg) Put a char, Nn ints
;
;      export _putch
_putch:
wt_putch:
#if SIMULATE
#elseif
        sta copctl                ; strobe Watch dog timer
        brclr SCTE,scs1,wt_putch     ;TDRE = 1 ?
        sta scdr                     ;OK, send
#endif
      rts
;
;     _pwordx(x-reg) Output 16 bit number pointed to by h:x-reg as 4 ascii chars
;
;      export _pwordx
_pwordx:
      lda ,x
      bsr _pbyte
      lda 1,x
      bsr _pbyte
      rts
;
;     _pword(x-reg) Output 16 bit number in h:x-reg as 4 ascii chars
;
;      export _pword
_pword:
      pshhx
      tha
      bsr _pbyte
      txa
      bsr _pbyte
      pulhx
      rts
;
;     _pbyte_sp(unsigned char b-reg) Output Ascii encoded byte, No int.
;                                    and a space
;
;      export _pbyte_sp
_pbyte_sp:
      bsr _space
      bra _pbyte
;
;     pbytex  Put a byte pointed by x-reg
;
;      export _pbytex
_pbytex:
      lda ,x
;
;     _pbyte(unsigned char a-reg) Output Ascii encoded byte, No int.
;
;      export _pbyte
_pbyte:
      pshhx
      jsr _dblchar
      tha
      bsr _putch
      txa
      bsr _putch
      pulhx
      rts
;
;     Checks for char, returns it in b-reg, =0 if none
;
;      export _ischar
_ischar:
      clra
      sta copctl
      brclr SCRF,scs1,no_char   ;SCRF = 1 ?
      lda scdr                  ;OK, get
no_char:
        rts
;
;     _getchne() get a character, no echo, returns in a-reg
;             returns carry set if ESC or CTRL-C
;
;        export _getchne
_getchne:
      sta copctl
      brclr SCRF,scs1,_getchne   ;SCRF = 1 ?
      lda scdr                   ;OK, get
      bra no_geco
;
;     _getch() get a character, echo, returns in a-reg
;             returns carry set if ESC or CTRL-C
;
;      export _getch
_getch:
      sta copctl
      brclr SCRF,scs1,_getch     ;SCRF = 1 ?
      lda scdr                   ;OK, get
      bsr _putch
no_geco:
      clc
      cmp #ESC
      beq gc_can
      cmp #$0C
      bne egchar
gc_can:
      sec
egchar:
      rts
;
;     _do_again  Returns z set(zero) if again, clear if end
;
;      export _do_again
_do_again:
      bsr _getch
      cmp #SPACE
      beq yep_again
      cmp #CR
      bne _do_again
      tsta
yep_again:
      rts
;
;     _puts(unsigned char *) Print a string passed pointed by d (a,b)
;       This subroutine will output the null teminated string
;       pointed to by H:X to the SCI.
;
;   Calling convention:
;
;       ldhx    #string
;       jsr     _puts
;
;   Returns:    nothing
;
;   Changes:    H:X
;
;
puts_lp:
        sta copctl                       ; strobe Watch dog timer
        brclr   SCTE,scs1,puts_lp        ; wait until SCI transmitter is empty
        mov     x+,scdr                  ; output character to the SCI and advance pointer
;      export _puts
_puts:
        tst     ,x                       ; test string character
        bne     puts_lp                  ; loop back if not null
        rts                              ; else, return
;
;   GetHexByte Subroutine  **************************************************
;
;       This subroutine retrieves two ASCII bytes via the SCI and converts
;       (packs) them into one hex byte, which is returned in ACC.
;
;   Calling convention:
;
;       jsr     GetHexByte
;
;   Returns:    CCRZ= 1 if valid hex byte retrieved.  Otherwise, CCRZ= 0.
;               ACC= data
;
;   Changes:    ACC
;
;      export GetHexByte
GetHexByte:
        bsr     _getchne          ; get msb character from the SCI
#if   HYPERTERM
#elseif
          jsr     _putch            ; echo it back
#endif
        bsr     _ishex             ; check if valid ASCII hex character
        beq     GetHexByte2       ; exit if not
        bsr     ToHex             ; convert ASCII hex character to hex value
        nsa                       ; swap lower nibble up
        psha                      ; save temporarily
        bsr     _getchne          ; get lsb character from the SCI
#if   HYPERTERM
#elseif
          jsr     _putch            ; echo it back
#endif
        bsr     _ishex            ; check if valid ASCII hex character
        beq     GetHexByte1       ; exit if not
        bsr     ToHex             ; convert ASCII hex character to hex value
        add     1,sp              ; combine msb and lsb nibbles
        bit     #0                ; CCRZ= 1
GetHexByte1:
        ais     #1                ; deallocate local variable
GetHexByte2:
        rts                       ; return
;
;     Look up address in table
;        Passed: byte to compare in a-reg
;                table address in h:x reg
;                   WARNING: last table entry must be null
;        Returns: Carry set if found, clear if not
;                 If found, returns value in address in h:x reg
;
;      export _get_atbl
_get_atbl:
      cmp ,x
      bne not_match
      lda 1,x
      psha
      lda 2,x
      tax
      pulh
      sec
      rts
not_match:
      aix #3
      tst ,x
      bne _get_atbl
      clc
      rts

;
;   ToHex Subroutine  *******************************************************
;
;       This subroutine converts the ASCII hex value passed in ACC to a
;       binary hex value.
;
;   Calling convention:
;
;       lda     data
;       jsr     ToHex
;
;   Returns:    ACC= data.
;
;   Changes:    ACC
;
;      export ToHex
ToHex:
        sub     #'0'              ; adjust first by subtracting '0'
        cmp     #9T                ; check if value was between '0' to '9'
        bls     ToHex1            ; exit if so
        sub     #7                ; else, adjust for value between 'A' to 'F'
ToHex1:
        rts                       ; return

;
;      (unsigned char reg-a)_ishex(unsigned char reg-a)
;                            Tests if it is an ascii hex
;                              Returns char if yes
;                              Returns 0 if no
;
;      export _ishex
_ishex:
      cmp #'0'
      blo nothex
      cmp #'9'
      bls yeshex
      and #$DF
      cmp #'F'
      bhi nothex
      cmp #'A'
      blo nothex
yeshex:
      tsta
      rts
nothex:
      clra
      rts
;
;     clr_spaces  Increments x:h pointer past spaces
;
;      export _clr_spaces
_clr_spaces:
      psha
cl_sp_lp:
      lda ,x
      cmp #SPACE
      bne not_sp
      aix #1
      bra cl_sp_lp
not_sp:
      pula
      rts
;
;   (unsigned int)_first_hex(unsigned char a-reg)
;                  Gets 1st parameter in buffer: 4 or a-reg hex chars
;   (unsigned int)_next_hex(unsigned char a-reg)
;                  Gets successive parameters in buffer: 4 or a-reg hex chars
;                  returns number in X-reg & carry clear if ok
;                     if error or cancel, set carry
;
;        export _next_hex
_next_hex:
        lda _hbufpos
        tah
        ldx _hbufpos+1
        incx
        bra _again_hex
;        export _first_hex
_first_hex:
        ldhx #_inbuf+1
_again_hex:
        bsr _clr_spaces
        bra _readhex
;
;   (unsigned int)_gethex(unsigned char a-reg)
;                  Gets up to 4  or a-reg length digit number,
;                  returns number in X-reg & carry clear if ok
;                     if error or cancel, set carry
;
;      export _gethex
_gethex:
      jsr _sp_gets
      bcs buf_can
      ldhx #_inbuf
;      export _readhex
_readhex:
      clra
      pshh
      pshx
numck_lp:
      psha
      lda ,x
      beq end_num_str
      cmp #SPACE
      beq end_num_str
      jsr _ishex
      pula
      beq err_can
      aix #1
      inca
      cmp #$05
      bne numck_lp
      bra err_can
end_num_str:
      tha
      sta _hbufpos
      stx _hbufpos+1
      pula
      tsta
      beq err_can
gnumok:
      pulx
      pulh
      cmp #$04
      beq gcv4
      cmp #$03
      beq gcv3
      cmp #$02
      beq gcv2
      bra gcv1
gcv4:
      pshx
      pshh
      lda  ,x
      ldx 1,x
      tah
gcv3_cut:
      jsr _ahex2b
      pulh
      pulx
      psha
      lda  2,x
      ldx 3,x
      tah
      jsr _ahex2b
      tax
      pulh
      bra gvc_end
gcv3:
      aix #-1
      pshx
      pshh
      lda #'0'
      ldx 1,x
      tah
      bra gcv3_cut
gcv2:
      lda  ,x
      ldx 1,x
      tah
      bra gcv1a
gcv1:
      ldhx #0
      ldx ,x
gcv1a:
      jsr _ahex2b
      ldhx #0
      tax
gvc_end:
      clc
      rts
err_can:
      pulx
      pulh
buf_can:
      sec
      rts
