;***************************************************************
;                                                              *
;  disasm.s  disassembler code                                 *
;                                                              *
;                                                              *
;**************************************************************/
;       include "./xref.inc"    ; Cosmic external references

;         include "./ab_regs.inc" ; MC68HC908AB32 register defimitions
;         include "./macros.inc"
;         include "./local.inc"

#if DEVEL
;        section .devtext
#elseif
;        section .text
#endif
;
#if DISASM
;
;     disasm  Disassembly 20 instructions
;             Passed address in h:x reg
;             return address in h:x incremented to next instruction
;
;      export _disasm
_disasm:
      jsr _crlf
      ldhx #_inbuf+1
      jsr _clr_spaces
      jsr _readhex
      bcs end_disasm
disasm_again:
      lda #20T
disasm_lp:
      bsr _disasm_1
      deca
      bne disasm_lp
end_disasm:
      jsr _do_again
      beq disasm_again
      rts
;
;     _disasm_1  Disassemble one instruction, pointed to by h:x reg
;
;      export _disasm_1x
_disasm_1x:
      psha
      jsr _space
      bra dia_gt
;      export _disasm_1
_disasm_1:
      psha
      jsr _pword            ; Print address
      jsr _space
dia_gt:
      jsr _get_type
      jsr _pr_op_bytes      ; Print op code bytes
      jsr _pr_operator      ; Print op code
      jsr _pr_operand       ; Print instuction operand
      jsr _inc_inst         ; Increment to next instruction
      jsr _crlf
      pula
      rts
;
;    _inc_inst  Increments to next instruction
;               Passed -n a-reg: Two low bits = # of bytes of intruction-1
;               clobbers a-reg
;
_inc_inst:
      and #$03
      inca
      aax
      rts
;
;     Print operator mnemonic
;
;              Passed h:x pointing to instruction
;              saves h:x, saves a-reg
;
_pr_operator:
      psha
      pshhx
      cmp #TY_INV      ; If invalid instruction, do that
      bne not_inv_ins
      ldhx #_inv_ins_mess
      bra out_op_code
not_inv_ins:
      lda ,x
      cmp #$1f          ; If greater than $1f, not a bit instruction
      bhi not_bit_ins   ; Map bit instructions: B0 = brset
      and #$11          ;                       B1 = brclr
      cmp #$10         ;                       B2 = bset
      blo bit_br        ;                       B3 = bclr
      and #$01
      ora #$02
bit_br:
      ora #$B0
      bra go_get_op
not_bit_ins:
      cmp #$9e
      bne not_dbl_op
      lda 1,x
not_dbl_op:
      cmp #$A7
      bne not_ais
      ldhx #_ais_ins
      bra out_op_code
not_ais:
      cmp #$AD
      bne not_bsr
      ldhx #_bsr_ins
      bra out_op_code
not_bsr:
      cmp #$AF
      bne not_aix
      ldhx #_aix_ins
      bra out_op_code
not_aix:
      blo go_get_op
      and #$0F
      ora #$A0
go_get_op:
      sub #$20          ; Table begins at Hex 20
      jsr _get_op_string
out_op_code:
      lda #3
pr_op_lp:
      psha
      lda ,x
      cmp #'a'
      blt _spec_let
      jsr _putch
      aix #1
      pula
      deca
      bne pr_op_lp
      jsr _space
end_spec4:
      jsr _space
end_pr_op:
      pulhx
      pula
      rts
;
;
_spec_let:
      pulx                 ; re-adjust stack
      cmp #'@'
      blt _spec_let5
      sub #'@'             ; Subtract $40 (char @) to get offset in table
      asla                 ; Multiply offset by 2
      tax
      clrh
      lda _char_tbl_4,x    ; Print 2 charcaters from table
      jsr _putch
      lda _char_tbl_4+1,x
      jsr _putch
      bra end_spec4
_spec_let5:
      sub #'1'             ; Subtract $31 (char 1) to get offset in table
      psha
      asla                 ; Multiply offset by 3
      add 1,sp
      pulx
      tax
      clrh
      lda _char_tbl_5,x    ; Print 3 charcaters from table
      jsr _putch
      lda _char_tbl_5+1,x
      jsr _putch
      lda _char_tbl_5+2,x
      jsr _putch
      bra end_pr_op
;
;     _get_type  Gets instruction type
;                      Passed instruction pointed to by h:x
;                      Two low bits = # of bytes of intruction-1
;                      Six hi bits = instruction type times 4
;
_get_type:
      pshhx
      lda ,x
      cmp #$9E
      bne not_ext_type
      lda 1,x
      ldhx #_ext_6_tbl
      cmp #$6f
      bhi gt_d0_ef
      and #$0f
      bra get_ty_ofst
gt_d0_ef:
      cmp #$df
      bhi gt_e0_ef
      and #$0f
      ora #$10
      bra get_ty_ofst
gt_e0_ef:
      and #$0f
      ora #$20
      bra get_ty_ofst
not_ext_type:
      ldhx #_type_tbl
get_ty_ofst:
      aax
      lda ,x
      pulhx
      rts
;
;     _pr_op_bytes Print op code bytes
;                   Passed: # of op code bytes -1 in a-reg, bits 0:1
;                   Passed: h:x pointing at instruction
;                   saves h:x, saves a-reg
;
_pr_op_bytes:
      pshhx
      psha
      and #$03
      inca
      psha
pr_ob_lp:
      psha
      lda ,x
      jsr _pbyte
      jsr _space
      aix #1
      pula
      deca
      bne pr_ob_lp
      pula
pr_sp_lp:
      cmp #5
      bhs done_pr_op
      psha
      jsr _space
      jsr _space
      jsr _space
      pula
      inca
      bra pr_sp_lp
done_pr_op:
      pula
      pulhx
      rts
;
;     _get_op_string:
;
;     Look up instruction in table
;        Passed: offset byte to compare in a-reg
;        Return: Address of op code on H:x reg
;        areg is clobbered
;
_get_op_string:
      ldx #3                 ; Do 3 bytes offset
      pshx                   ; Put counter on stack
      ldhx #_dis_tbl         ; Init h:x to beginning of table
get_op_lp:
      psha                   ; Save op code
      pshx
      add 1,sp               ; add to low byte
      pulx
      tax
      bcc not_cary
      pshh                   ; If carry, increment hi byte
      inc 1,sp
      pulh
not_cary:
      pula                   ;restore op code
      dbnz 1,sp,get_op_lp
      pula
      rts
;
;     _pr_operand
;        passed: operator type in a-reg
;                operator address in b-reg
;
_pr_operand:
      psha
      pshhx
      jsr _space ; print space before operand
      asra
      asra       ; Put type in lower bits, clear out byte count bits
      psha       ; save type
      lda ,x
      cmp #$9e           ; If two byte instruction, increment pointer
      bne not_2byte
      aix #1
not_2byte:
      lda #1
      cmp 1,sp            ; If type = 0 or 1, do brset/brclr operand
      bmi ck_bitsc
      jsr _pr_bit
      jsr _pr_comma
      lda ,x              ; rrr micom fix: not printing operand
      jsr _pbyte          ; rrr
      jsr _space          ; rrr
      aix #1              ; rrr
      jsr _pr_relative2   ; rrr
      bra end_popy
ck_bitsc:
      inca
      inca
      cmp 1,sp            ; If type = 2 or 3, do bset/bclr operand
      bmi ck_branch
      jsr _pr_bit
      jsr _pr_comma       ; rrr micom fix: not printing operand
      lda ,x              ; rrr
      jsr _pbyte          ; rrr
      bra end_popy
ck_branch:
      aix #1
      inca
      cmp 1,sp            ;If Type branch, print destination address
      bne ck_inh
      jsr _pr_relative
      bra end_popy
ck_inh:
      inca
      cmp 1,sp            ; If Inherent, nothing to do
      bne ck_im1
      bra end_popy
ck_im1:
      inca
      cmp 1,sp            ; If Immediate one byte, print byte
      bne ck_im2
      jsr _pr_pound
      jsr _pbytex
end_popy:
      bra end_popx
ck_im2:
      inca
      cmp 1,sp            ; If Immediate two bytes, print word
      bne ck_dir
      jsr _pr_pound
      bra do_pwordx
ck_dir:
      inca
      cmp 1,sp            ; If direct page, print 1 byte address
      bne ck_ext
      jsr _pbytex
      bra end_popx
ck_ext:
      inca
      cmp 1,sp             ; If extended, print 2 byte address
      bne ck_ix0
do_pwordx:
      jsr _pwordx
      bra end_popx
ck_ix0:
      inca
      cmp 1,sp             ; If indexed, no offset, print ,x
      bne ck_ix1
out_x:
      jsr _pr_comma
      lda #'x'
      jsr _putch
      bra end_popx
ck_ix1:
      inca
      cmp 1,sp            ; If indexed, one byte offset, print byte,x
      bne ck_ix2
      jsr _pbytex
      bra out_x
ck_ix2:
      inca
      cmp 1,sp            ; If indexed, two byte offset, print word,x
      bne ck_sp1
      jsr _pwordx
      bra out_x
ck_sp1:
      inca
      cmp 1,sp            ; If stack pointer, one byte offset, print byte,sp
      bne ck_sp2
      jsr _pbytex
      bra do_comma_sp
ck_sp2:
      inca
      cmp 1,sp            ; If stack pointer, two byte offset, print word,sp
      bne ck_mid
      jsr _pwordx
do_comma_sp:
      jsr _comma_sp
      bra end_popx
ck_mid:
      inca
      cmp 1,sp            ; If mov imm to dir, print #imm_byte,dir_byte
      bne ck_mdd
      jsr _pr_pound
out_mov:
      jsr _pbytex
      aix #1
      jsr _pr_comma
      jsr _pbytex
end_popx:
      bra end_pop
ck_mdd:
      inca
      cmp 1,sp            ; If mov dir to dir, print dir_byte,dir_byte
      beq out_mov
ck_idp:
      inca
      cmp 1,sp            ; If mov indexed to dir, post inc, print x+,dir_byte
      bne ck_dip
      bsr pr_xplus
      jsr _pr_comma
      jsr _pbytex
      bra end_pop
ck_dip:
      inca
      cmp 1,sp            ; If mem to direct, post inc, print dir_byte,x+
      bne ck_cbx
      jsr _pbytex
      jsr _pr_comma
      bsr pr_xplus
      bra end_pop
ck_cbx:
      inca
      cmp 1,sp            ; If cmp-br, index, post increment, print x+,addr
      bne ck_cb1
cbr_x:
      bsr pr_xplus
cbr_rel0:
      jsr _pr_comma
      bsr _pr_relative
      bra end_pop
ck_cb1:
      inca
      cmp 1,sp            ; If cmp-br, index, 8 ofst, post+, print byte,x+,addr
      bne ck_cpd
      jsr _pr_pound
      jsr _pbytex
      aix #1
      bra cbr_x
ck_cpd:
      inca
      cmp 1,sp            ; If cmp-branch, direct-relative, print byte,addr
      bne ck_cpi
cbr_rel1:
      jsr _pbytex
      aix #1
      jsr _pr_comma
      bsr _pr_relative2
      bra end_pop
ck_cpi:
      inca
      cmp 1,sp            ; If cmp-branch, immediate-relative, print #byte,addr
      bne ck_cbs
      bra cbr_rel1
ck_cbs:
      inca
      cmp 1,sp            ;  If cmp-br, direct sp relative, print byte,sp,addr
      bne end_pop
      jsr _pbytex
      aix #1
      bra cbr_rel0
end_pop:
      pula
      pulhx
      pula
      rts
;
;     _pr_bit  prints bit number for bit instructions
;              and increments h:x reg to next byte
;
_pr_bit:
      lda ,x
      rora
      and #$07
      jsr _pr_nibble
      aix #1
      rts
;
;     Print x+
;
pr_xplus:
      jsr _pr_letter_x
      lda #'+'
      jsr _putch
      rts
;
;     _pr_relative   Passed: h:x reg pointed to realative branch value
;
;
_pr_relative2:
      lda 1,x
      aix #1
      bra do_pr_rel
_pr_relative:
      lda ,x
do_pr_rel:
      bmi branch_back
      aix #1
rel_fw_lp:
      aix #1
      deca
      bne rel_fw_lp
      bra out_rel_addr
branch_back:
      inca
rel_bk_lp:
      aix #-1
      inca
      bne rel_bk_lp
out_rel_addr:
      jsr _pword
      rts
;
;     Oddball bsr op code
;
_bsr_ins:
      fcb "bsr"
;
;     Oddball ais op code
;
_ais_ins:
      fcb "ais"
;
;     Oddball aix op code
;
_aix_ins:
      fcb "aix"
;
;   Dissassembler lookup table
;
_dis_tbl:
      ; Instruction Hex: $20
      fcb "bra"
      ; Instruction Hex: $21
      fcb "brn"
      ; Instruction Hex: $22
      fcb "bhi"
      ; Instruction Hex: $23
      fcb "bls"
      ; Instruction Hex: $24
      fcb "bcc"
      ; Instruction Hex: $25
      fcb "bcs"
      ; Instruction Hex: $26
      fcb "bne"
      ; Instruction Hex: $27
      fcb "beq"
      ; Instruction Hex: $28
      fcb "bhA"               ; ***  bhcc Map A to cc
      ; Instruction Hex: $29
      fcb "bhB"               ; ***  bhcs  Map B to cs
      ; Instruction Hex: $2A
      fcb "bpl"
      ; Instruction Hex: $2B
      fcb "bmi"
      ; Instruction Hex: $2C
      fcb "bmc"
      ; Instruction Hex: $2D
      fcb "bms"
      ; Instruction Hex: $2E
      fcb "bil"
      ; Instruction Hex: $2F
      fcb "bih"
      ; Instruction Hex: $30
      fcb "neg"
      ; Instruction Hex: $31
      fcb "cbC"               ; *** cbeq  Map C to eq
      ; Instruction Hex: $32
      fcb "err"
      ; Instruction Hex: $33
      fcb "com"
      ; Instruction Hex: $34
      fcb "lsr"
      ; Instruction Hex: $35
      fcb "stD"               ; *** sthx  Map D to hx
      ; Instruction Hex: $36
      fcb "ror"
      ; Instruction Hex: $37
      fcb "asr"
      ; Instruction Hex: $38
      fcb "lsl"
      ; Instruction Hex: $39
      fcb "rol"
      ; Instruction Hex: $3A
      fcb "dec"
      ; Instruction Hex: $3B
      fcb "dbQ"               ; *** dbnz  Map Q to nz
      ; Instruction Hex: $3C
      fcb "inc"
      ; Instruction Hex: $3D
      fcb "tst"
      ; Instruction Hex: $3E
      fcb "err"
      ; Instruction Hex: $3F
      fcb "clr"
      ; Instruction Hex: $40
      fcb "neY"               ; *** nega  Map Y to ga
      ; Instruction Hex: $41
      fcb "cb6"               ; *** cbeqa Map 6 to eqa
      ; Instruction Hex: $42
      fcb "mul"
      ; Instruction Hex: $43
      fcb "coZ"               ; *** coma  Map Z to ma
      ; Instruction Hex: $44
      fcb "lsE"               ; *** lsra  Map E to ra
      ; Instruction Hex: $45
      fcb "ldD"               ; *** ldhx  Map D to hx
      ; Instruction Hex: $46
      fcb "roE"               ; *** rora  Map E to ra
      ; Instruction Hex: $47
      fcb "asE"               ; *** asra  Map E to ra
      ; Instruction Hex: $48
      fcb "lsF"               ; *** lsla  Map F to la
      ; Instruction Hex: $49
      fcb "roF"               ; *** rola  Map F to la
      ; Instruction Hex: $4A
      fcb "deG"               ; *** deca  Map G to ca
      ; Instruction Hex: $4B
      fcb "db1"               ; *** dbnza Map 1 to nza
      ; Instruction Hex: $4C
      fcb "inH"               ; *** inca  Map H to ca
      ; Instruction Hex: $4D
      fcb "tsI"               ; *** tsta  Map I to ta
      ; Instruction Hex: $4E
      fcb "mov"
      ; Instruction Hex: $4F
      fcb "clE"               ; *** clra  Map E to ra
      ; Instruction Hex: $50
      fcb "neJ"               ; *** negx  Map J to gx
      ; Instruction Hex: $51
      fcb "cb2"               ; *** cbeqx Map 2 to eqx
      ; Instruction Hex: $52
      fcb "div"
      ; Instruction Hex: $53
      fcb "coK"               ; *** comx  Map K to mx
      ; Instruction Hex: $54
      fcb "lsL"               ; *** lsrx  Map L to rx
      ; Instruction Hex: $55
      fcb "ldD"               ; *** ldhx  Map D to hx
      ; Instruction Hex: $56
      fcb "roL"               ; *** rorx  Map L to rx
      ; Instruction Hex: $57
      fcb "asM"               ; *** asrx  Map M to rx
      ; Instruction Hex: $58
      fcb "lsN"               ; *** lslx  Map N to lx
      ; Instruction Hex: $59
      fcb "roN"               ; *** rolx  Map N to lx
      ; Instruction Hex: $5A
      fcb "deO"               ; *** decx  Map O to cx
      ; Instruction Hex: $5B
      fcb "db3"               ; *** dbnzx Map 3 to nzx
      ; Instruction Hex: $5C
      fcb "inO"               ; *** incx  Map O to cx
      ; Instruction Hex: $5D
      fcb "tsP"               ; *** tstx  Map P to tx
      ; Instruction Hex: $5E
      fcb "mov"
      ; Instruction Hex: $5F
      fcb "clL"               ; *** clrx  Map L to rx
      ; Instruction Hex: $60
      fcb "neg"
      ; Instruction Hex: $61
      fcb "cbC"               ; *** cbeq  Map C to eq
      ; Instruction Hex: $62
      fcb "nsa"
      ; Instruction Hex: $63
      fcb "com"
      ; Instruction Hex: $64
      fcb "lsr"
      ; Instruction Hex: $65
      fcb "cpD"               ; *** cphx  Map D to hx
      ; Instruction Hex: $66
      fcb "ror"
      ; Instruction Hex: $67
      fcb "asr"
      ; Instruction Hex: $68
      fcb "lsl"
      ; Instruction Hex: $69
      fcb "rol"
      ; Instruction Hex: $6A
      fcb "dec"
      ; Instruction Hex: $6B
      fcb "dbQ"               ; *** dbnz  Map Q to nz
      ; Instruction Hex: $6C
      fcb "inc"
      ; Instruction Hex: $6D
      fcb "tst"
      ; Instruction Hex: $6E
      fcb "mov"
      ; Instruction Hex: $6F
      fcb "clr"
      ; Instruction Hex: $70
      fcb "neg"
      ; Instruction Hex: $71
      fcb "cbC"               ; *** cbeq  Map C to eq
      ; Instruction Hex: $72
      fcb "daa"
      ; Instruction Hex: $73
      fcb "com"
      ; Instruction Hex: $74
      fcb "lsr"
      ; Instruction Hex: $75
      fcb "cpD"               ; *** cphx  Map D to hx
      ; Instruction Hex: $76
      fcb "ror"
      ; Instruction Hex: $77
      fcb "asr"
      ; Instruction Hex: $78
      fcb "lsl"
      ; Instruction Hex: $79
      fcb "rol"
      ; Instruction Hex: $7A
      fcb "dec"
      ; Instruction Hex: $7B
      fcb "dbQ"               ; *** dbnz  Map Q to nz
      ; Instruction Hex: $7C
      fcb "inc"
      ; Instruction Hex: $7D
      fcb "tst"
      ; Instruction Hex: $7E
      fcb "mov"
      ; Instruction Hex: $7F
      fcb "clr"
      ; Instruction Hex: $80
      fcb "rti"
      ; Instruction Hex: $81
      fcb "rts"
      ; Instruction Hex: $82
      fcb "err"
      ; Instruction Hex: $83
      fcb "swi"
      ; Instruction Hex: $84
      fcb "tap"
      ; Instruction Hex: $85
      fcb "tpa"
      ; Instruction Hex: $86
      fcb "puF"               ; *** pula  Map F to la
      ; Instruction Hex: $87
      fcb "psR"               ; *** psha  Map R to ha
      ; Instruction Hex: $88
      fcb "puN"               ; *** pulx  Map N to lx
      ; Instruction Hex: $89
      fcb "psD"               ; *** pshx  Map D to hx
      ; Instruction Hex: $8A
      fcb "puS"               ; *** pulh  Map S to lh
      ; Instruction Hex: $8B
      fcb "psT"               ; *** pshh  Map T to hh
      ; Instruction Hex: $8C
      fcb "clU"               ; *** clrh  Map U to rh
      ; Instruction Hex: $8D
_inv_ins_mess:
      fcb "err"
      ; Instruction Hex: $8E
      fcb "stV"               ; *** stop  Map V to op
      ; Instruction Hex: $8F
      fcb "wa@"               ; *** wait  Map @ to it
      ; Instruction Hex: $90
      fcb "bge"
      ; Instruction Hex: $91
      fcb "blt"
      ; Instruction Hex: $92
      fcb "bgt"
      ; Instruction Hex: $93
      fcb "ble"
      ; Instruction Hex: $94
      fcb "txs"
      ; Instruction Hex: $95
      fcb "tsx"
      ; Instruction Hex: $96
      fcb "err"
      ; Instruction Hex: $97
      fcb "tax"
      ; Instruction Hex: $98
      fcb "clc"
      ; Instruction Hex: $99
      fcb "sec"
      ; Instruction Hex: $9A
      fcb "cli"
      ; Instruction Hex: $9B
      fcb "sei"
      ; Instruction Hex: $9C
      fcb "rsp"
      ; Instruction Hex: $9D
      fcb "nop"
      ; Instruction Hex: $9E
      fcb "err"
      ; Instruction Hex: $9F
      fcb "txa"
      ; Instruction Hex: $A0
      fcb "sub"
      ; Instruction Hex: $A1
      fcb "cmp"
      ; Instruction Hex: $A2
      fcb "sbc"
      ; Instruction Hex: $A3
      fcb "cpx"
      ; Instruction Hex: $A4
      fcb "and"
      ; Instruction Hex: $A5
      fcb "bit"
      ; Instruction Hex: $A6
      fcb "lda"
      ; Instruction Hex: $A7
      fcb "sta"
      ; Instruction Hex: $A8
      fcb "eor"
      ; Instruction Hex: $A9
      fcb "adc"
      ; Instruction Hex: $AA
      fcb "ora"
      ; Instruction Hex: $AB
      fcb "add"
      ; Instruction Hex: $AC
      fcb "jmp"
      ; Instruction Hex: $AD
      fcb "jsr"
      ; Instruction Hex: $AE
      fcb "ldx"
      ; Instruction Hex: $AF
      fcb "stx"
;
;    Bit Instruction lookup table addon
;
      ; Special Intruction, 00 mapped to B0
      fcb "br4"               ; *** brset Map 4 to set
      ; Special Intruction, 01 mapped to B1
      fcb "br5"               ; *** brclr Map 5 to clr
      ; Special Intruction, 10 mapped to B2
      fcb "bsW"               ; *** bset  Map W to et
      ; Special Intruction, 11 mapped to B3
      fcb "bcX"               ; *** bclr  Map X to lr

;
;     Map for op codes of length 4
;
_char_tbl_4:
      fcb "it"               ;  Map @ to it wait
      fcb "cc"               ;  Map A to cc bhcc
      fcb "cs"               ;  Map B to cs bhcs
      fcb "eq"               ;  Map C to eq cbeq
      fcb "hx"               ;  Map D to hx sthx ldhx cphx pshx
      fcb "ra"               ;  Map E to ra lsra rora asra clra
      fcb "la"               ;  Map F to la lsla rola pula
      fcb "ca"               ;  Map G to ca deca
      fcb "ca"               ;  Map H to ca inca
      fcb "ta"               ;  Map I to ta tsta
      fcb "gx"               ;  Map J to gx negx
      fcb "mx"               ;  Map K to mx comx
      fcb "rx"               ;  Map L to rx lsrx clrx rorx
      fcb "rx"               ;  Map M to rx asrx
      fcb "lx"               ;  Map N to lx lslx rolx pulx
      fcb "cx"               ;  Map O to cx decx incx
      fcb "tx"               ;  Map P to tx tstx
      fcb "nz"               ;  Map Q to nz dbnz
      fcb "ha"               ;  Map R to ha psha
      fcb "lh"               ;  Map S to lh pulh
      fcb "hh"               ;  Map T to hh pshh
      fcb "rh"               ;  Map U to rh clrh
      fcb "op"               ;  Map V to op stop
      fcb "et"               ;  Map W to et bset
      fcb "lr"               ;  Map X to lr bclr
      fcb "ga"               ;  Map Y to ga nega
      fcb "ma"               ;  Map Z to ma coma
;
;     Map for op codes of length 5
;
_char_tbl_5:
      fcb "nza"              ;  Map 1 to nza dbnza
      fcb "eqx"              ;  Map 2 to eqx cbeqx
      fcb "nzx"              ;  Map 3 to nzx dbnzx
      fcb "set"              ;  Map 4 to set brset
      fcb "clr"              ;  Map 5 to clr brclr
      fcb "eqa"              ;  Map 6 to eqa cbeqa
;
;     Instruction Types
;
TY_BRSET  equ 0*4   ; Test bit set and branch type instruction
TY_BRCLR  equ 1*4   ; Test bit clear and branch type instruction
TY_BSET   equ 2*4   ; Set Bit type instruction
TY_BCLR   equ 3*4   ; Clear Bit type instruction
TY_REL    equ 4*4   ; Relative, branch type instuction, ie BRA, BHI ...
TY_INH    equ 5*4   ; Inherent instruction, ie SWI, RTS, CLRA, etc.
TY_IM1    equ 6*4   ; Immediate 1 byte instructions, ie lda #$12
TY_IM2    equ 7*4   ; Immediate 2 byte instructions, ie ldhx #$1234
TY_DIR    equ 8*4   ; Direct page, instructions accesss first 256 bytes
TY_EXT    equ 9*4   ; Extended, instructions using 16 bit addresses
TY_IX0    equ 10*4  ; Indexed, no offset, ie. lda ,X
TY_IX1    equ 11*4  ; Indexed, 1 byte offset, ie. lda 5,x
TY_IX2    equ 12*4  ; Indexed, 1 byte offset, ie. lda 5,x
TY_SP1    equ 13*4  ; Stack Pointer, 1 byte offset, ie. lda 5,sp
TY_SP2    equ 14*4  ; Stack Pointer, 2 byte offset, ie. lda $1235,sp
TY_MID    equ 15*4  ; Memory to Memory Immediate to Direct, ie mov #3,$50
TY_MDD    equ 16*4  ; Memory to Memory Direct to Direct, ie mov $51,$50
TY_IDP    equ 17*4  ; Memory to Memory Indexed to Direct post inc, mov x+,$50
TY_DIP    equ 18*4  ; Memory to Memory Direct to Indexed post inc, mov $50,x+
TY_CBX    equ 19*4  ; Compare branch, indexed post inc, ie cbeq x+,LABEL
TY_CB1    equ 20*4  ; Cmp branch, index w/8 bit ofst post inc, cbeq $5,x+,LABEL
TY_CBD    equ 21*4  ; Compare branch direct relative, ie cbeq $50,LABEL
TY_CBI    equ 22*4  ; Compare branch immediate relative, ie cbeq #5,LABEL
TY_CBS    equ 23*4  ; Compare branch direct sp relative, cbeq $50,sp,LABEL
TY_INV    equ 24*4  ; Invalid
;
;    Instruction type table
;
_type_tbl:
  ;   Type(hi 6 bits) or'd with #bytes-1 (low 2 bits)
      fcb (TY_BRSET+3-1)         ; 00 brset 0
      fcb (TY_BRCLR+3-1)         ; 01 brclr 0
      fcb (TY_BRSET+3-1)         ; 02 brset 1
      fcb (TY_BRCLR+3-1)         ; 03 brclr 1
      fcb (TY_BRSET+3-1)         ; 04 brset 2
      fcb (TY_BRCLR+3-1)         ; 05 brclr 2
      fcb (TY_BRSET+3-1)         ; 06 brset 3
      fcb (TY_BRCLR+3-1)         ; 07 brclr 3
      fcb (TY_BRSET+3-1)         ; 08 brset 4
      fcb (TY_BRCLR+3-1)         ; 09 brclr 4
      fcb (TY_BRSET+3-1)         ; 0a brset 5
      fcb (TY_BRCLR+3-1)         ; 0b brclr 5
      fcb (TY_BRSET+3-1)         ; 0c brset 6
      fcb (TY_BRCLR+3-1)         ; 0d brclr 6
      fcb (TY_BRSET+3-1)         ; 0e brset 7
      fcb (TY_BRCLR+3-1)         ; 0f brclr 7
;
      fcb (TY_BSET+2-1)          ; 10 bset 0
      fcb (TY_BCLR+2-1)          ; 11 bclr 0
      fcb (TY_BSET+2-1)          ; 12 bset 1
      fcb (TY_BCLR+2-1)          ; 13 bclr 1
      fcb (TY_BSET+2-1)          ; 14 bset 2
      fcb (TY_BCLR+2-1)          ; 15 bclr 2
      fcb (TY_BSET+2-1)          ; 16 bset 3
      fcb (TY_BCLR+2-1)          ; 17 bclr 3
      fcb (TY_BSET+2-1)          ; 18 bset 4
      fcb (TY_BCLR+2-1)          ; 19 bclr 4
      fcb (TY_BSET+2-1)          ; 1a bset 5
      fcb (TY_BCLR+2-1)          ; 1b bclr 5
      fcb (TY_BSET+2-1)          ; 1c bset 6
      fcb (TY_BCLR+2-1)          ; 1d bclr 6
      fcb (TY_BSET+2-1)          ; 1e bset 7
      fcb (TY_BCLR+2-1)          ; 1f bclr 7
;
      fcb (TY_REL+2-1)           ; 20 bra
      fcb (TY_REL+2-1)           ; 21 brn
      fcb (TY_REL+2-1)           ; 22 bhi
      fcb (TY_REL+2-1)           ; 23 bls
      fcb (TY_REL+2-1)           ; 24 bcc
      fcb (TY_REL+2-1)           ; 25 bcs
      fcb (TY_REL+2-1)           ; 26 bne
      fcb (TY_REL+2-1)           ; 27 beq
      fcb (TY_REL+2-1)           ; 28 bhcc
      fcb (TY_REL+2-1)           ; 29 bhcs
      fcb (TY_REL+2-1)           ; 2a bpl
      fcb (TY_REL+2-1)           ; 2b bmi
      fcb (TY_REL+2-1)           ; 2c bmc
      fcb (TY_REL+2-1)           ; 2d bms
      fcb (TY_REL+2-1)           ; 2e bil
      fcb (TY_REL+2-1)           ; 2f bih
;
      fcb (TY_DIR+2-1)           ; 30 neg
      fcb (TY_CBD+3-1)           ; 31 cbeq
      fcb (TY_INV+1-1)           ; 32 invalid
      fcb (TY_DIR+2-1)           ; 33 com
      fcb (TY_DIR+2-1)           ; 34 lsr
      fcb (TY_DIR+2-1)           ; 35 sthx
      fcb (TY_DIR+2-1)           ; 36 ror
      fcb (TY_DIR+2-1)           ; 37 asr
      fcb (TY_DIR+2-1)           ; 38 lsl
      fcb (TY_DIR+2-1)           ; 39 rol
      fcb (TY_DIR+2-1)           ; 3a dec
      fcb (TY_CBD+3-1)           ; 3b dbnz
      fcb (TY_DIR+2-1)           ; 3c inc    ; Bug Fix Don Wood 7/10/02
                                             ;  was TY_INH
      fcb (TY_DIR+2-1)           ; 3d tst
      fcb (TY_INV+1-1)           ; 3e invalid
      fcb (TY_DIR+2-1)           ; 3f clr
;
      fcb (TY_INH+1-1)            ; 40 nega
      fcb (TY_CBI+3-1)            ; 41 cbeqa
      fcb (TY_INH+1-1)            ; 42 mul
      fcb (TY_INH+1-1)            ; 43 coma
      fcb (TY_INH+1-1)            ; 44 lsra
      fcb (TY_IM2+3-1)            ; 45 ldhx
      fcb (TY_INH+1-1)            ; 46 rora
      fcb (TY_INH+1-1)            ; 47 asra
      fcb (TY_INH+1-1)            ; 48 lsla
      fcb (TY_INH+1-1)            ; 49 rola
      fcb (TY_INH+1-1)            ; 4a deca
      fcb (TY_REL+2-1)            ; 4b dbnza
      fcb (TY_INH+1-1)            ; 4c inca
      fcb (TY_INH+1-1)            ; 4d tsta
      fcb (TY_MDD+3-1)            ; 4e mov
      fcb (TY_INH+1-1)            ; 4f clra
;
      fcb (TY_INH+1-1)            ; 50 negx
      fcb (TY_REL+2-1)            ; 51 cbeqx
      fcb (TY_INH+1-1)            ; 52 div
      fcb (TY_INH+1-1)            ; 53 comx
      fcb (TY_INH+1-1)            ; 54 lsrx
      fcb (TY_DIR+2-1)            ; 55 ldhx
      fcb (TY_INH+1-1)            ; 56 rorx
      fcb (TY_INH+1-1)            ; 57 asrx
      fcb (TY_INH+1-1)            ; 58 lslx
      fcb (TY_INH+1-1)            ; 59 rolx
      fcb (TY_INH+1-1)            ; 5a decx
      fcb (TY_REL+2-1)            ; 5b dbnzx
      fcb (TY_INH+1-1)            ; 5c incx
      fcb (TY_INH+1-1)            ; 5d tstx
      fcb (TY_DIP+2-1)            ; 5e mov
      fcb (TY_INH+1-1)            ; 5f clrx
;
      fcb (TY_IX1+2-1)            ; 60 neg
      fcb (TY_CB1+3-1)            ; 61 cbeq
      fcb (TY_INH+1-1)            ; 62 nsa
      fcb (TY_IX1+2-1)            ; 63 com
      fcb (TY_IX1+2-1)            ; 64 lsr
      fcb (TY_IM2+3-1)            ; 65 cphx
      fcb (TY_IX1+2-1)            ; 66 ror
      fcb (TY_IX1+2-1)            ; 67 asr
      fcb (TY_IX1+2-1)            ; 68 lsl
      fcb (TY_IX1+2-1)            ; 69 rol
      fcb (TY_IX1+2-1)            ; 6a dec
      fcb (TY_REL+3-1)            ; 6b dbnz
      fcb (TY_IX1+2-1)            ; 6c inc
      fcb (TY_IX1+2-1)            ; 6d tst
      fcb (TY_MID+3-1)            ; 6e mov
      fcb (TY_IX1+2-1)            ; 6f clr
;
      fcb (TY_IX0+2-1)            ; 70 neg
      fcb (TY_CBX+3-1)            ; 71 cbeq
      fcb (TY_INH+1-1)            ; 72 daa
      fcb (TY_IX0+1-1)            ; 73 com
      fcb (TY_IX0+1-1)            ; 74 lsr
      fcb (TY_DIR+2-1)            ; 75 cphx        ; Micom Fix
      fcb (TY_IX0+1-1)            ; 76 ror
      fcb (TY_IX0+1-1)            ; 77 asr
      fcb (TY_IX0+1-1)            ; 78 lsl
      fcb (TY_IX0+1-1)            ; 79 rol
      fcb (TY_IX0+1-1)            ; 7a dec
      fcb (TY_REL+2-1)            ; 7b dbnz
      fcb (TY_IX0+1-1)            ; 7c inc
      fcb (TY_IX0+1-1)            ; 7d tst
      fcb (TY_IDP+2-1)            ; 7e mov
      fcb (TY_IX0+1-1)            ; 7f clr
;
      fcb (TY_INH+1-1)            ; 80 rti
      fcb (TY_INH+1-1)            ; 81 rts
      fcb (TY_INV+1-1)            ; 82 invalid
      fcb (TY_INH+1-1)            ; 83 swi
      fcb (TY_INH+1-1)            ; 84 tap
      fcb (TY_INH+1-1)            ; 85 tpa
      fcb (TY_INH+1-1)            ; 86 pula
      fcb (TY_INH+1-1)            ; 87 psha
      fcb (TY_INH+1-1)            ; 88 pulx
      fcb (TY_INH+1-1)            ; 89 pshx
      fcb (TY_INH+1-1)            ; 8a pulh
      fcb (TY_INH+1-1)            ; 8b pshh
      fcb (TY_INH+1-1)            ; 8c clrh
      fcb (TY_INV+1-1)            ; 8d invalid
      fcb (TY_INH+1-1)            ; 8e stop
      fcb (TY_INH+1-1)            ; 8f wait
;
      fcb (TY_REL+2-1)            ; 90 bge
      fcb (TY_REL+2-1)            ; 91 blt
      fcb (TY_REL+2-1)            ; 92 bgt
      fcb (TY_REL+2-1)            ; 93 ble
      fcb (TY_INH+1-1)            ; 94 txs
      fcb (TY_INH+1-1)            ; 95 tsx
      fcb (TY_INH+1-1)            ; 96 invalid
      fcb (TY_INH+1-1)            ; 97 tax
      fcb (TY_INH+1-1)            ; 98 clc
      fcb (TY_INH+1-1)            ; 99 sec
      fcb (TY_INH+1-1)            ; 9a cli
      fcb (TY_INH+1-1)            ; 9b sei
      fcb (TY_INH+1-1)            ; 9c rsp
      fcb (TY_INH+1-1)            ; 9d nop
      fcb (TY_INV+1-1)            ; 9e pre-byte (invalid)
      fcb (TY_INH+1-1)            ; 9f wait
;
      fcb (TY_IM1+2-1)            ; a0 sub
      fcb (TY_IM1+2-1)            ; a1 cmp
      fcb (TY_IM1+2-1)            ; a2 sbc
      fcb (TY_IM1+2-1)            ; a3 cpx
      fcb (TY_IM1+2-1)            ; a4 and
      fcb (TY_IM1+2-1)            ; a5 bit
      fcb (TY_IM1+2-1)            ; a6 lda
      fcb (TY_IM1+2-1)            ; a7 ais
      fcb (TY_IM1+2-1)            ; a8 eor
      fcb (TY_IM1+2-1)            ; a9 adc
      fcb (TY_IM1+2-1)            ; aa ora
      fcb (TY_IM1+2-1)            ; ab add
      fcb (TY_INV+2-1)            ; ac invalid
      fcb (TY_REL+2-1)            ; ad bsr
      fcb (TY_IM1+2-1)            ; ae ldx
      fcb (TY_IM1+2-1)            ; af aix
;
      fcb (TY_DIR+2-1)            ; b0 sub
      fcb (TY_DIR+2-1)            ; b1 cmp
      fcb (TY_DIR+2-1)            ; b2 sbc
      fcb (TY_DIR+2-1)            ; b3 cpx
      fcb (TY_DIR+2-1)            ; b4 and
      fcb (TY_DIR+2-1)            ; b5 bit
      fcb (TY_DIR+2-1)            ; b6 lda
      fcb (TY_DIR+2-1)            ; b7 ais
      fcb (TY_DIR+2-1)            ; b8 eor
      fcb (TY_DIR+2-1)            ; b9 adc
      fcb (TY_DIR+2-1)            ; ba ora
      fcb (TY_DIR+2-1)            ; bb add
      fcb (TY_DIR+2-1)            ; bc jmp
      fcb (TY_DIR+2-1)            ; bd jsr
      fcb (TY_DIR+2-1)            ; be ldx
      fcb (TY_DIR+2-1)            ; bf stx
;
      fcb (TY_EXT+3-1)            ; c0 sub
      fcb (TY_EXT+3-1)            ; c1 cmp
      fcb (TY_EXT+3-1)            ; c2 sbc
      fcb (TY_EXT+3-1)            ; c3 cpx
      fcb (TY_EXT+3-1)            ; c4 and
      fcb (TY_EXT+3-1)            ; c5 bit
      fcb (TY_EXT+3-1)            ; c6 lda
      fcb (TY_EXT+3-1)            ; c7 ais
      fcb (TY_EXT+3-1)            ; c8 eor
      fcb (TY_EXT+3-1)            ; c9 adc
      fcb (TY_EXT+3-1)            ; ca ora
      fcb (TY_EXT+3-1)            ; cb add
      fcb (TY_EXT+3-1)            ; cc jmp
      fcb (TY_EXT+3-1)            ; cd jsr
      fcb (TY_EXT+3-1)            ; ce ldx
      fcb (TY_EXT+3-1)            ; cf stx
;
      fcb (TY_IX2+3-1)            ; d0 sub
      fcb (TY_IX2+3-1)            ; d1 cmp
      fcb (TY_IX2+3-1)            ; d2 sbc
      fcb (TY_IX2+3-1)            ; d3 cpx
      fcb (TY_IX2+3-1)            ; d4 and
      fcb (TY_IX2+3-1)            ; d5 bit
      fcb (TY_IX2+3-1)            ; d6 lda
      fcb (TY_IX2+3-1)            ; d7 ais
      fcb (TY_IX2+3-1)            ; d8 eor
      fcb (TY_IX2+3-1)            ; d9 adc
      fcb (TY_IX2+3-1)            ; da ora
      fcb (TY_IX2+3-1)            ; db add
      fcb (TY_IX2+3-1)            ; dc jmp
      fcb (TY_IX2+3-1)            ; dd jsr
      fcb (TY_IX2+3-1)            ; de ldx
      fcb (TY_IX2+3-1)            ; df stx
;
      fcb (TY_IX1+2-1)            ; e0 sub
      fcb (TY_IX1+2-1)            ; e1 cmp
      fcb (TY_IX1+2-1)            ; e2 sbc
      fcb (TY_IX1+2-1)            ; e3 cpx
      fcb (TY_IX1+2-1)            ; e4 and
      fcb (TY_IX1+2-1)            ; e5 bit
      fcb (TY_IX1+2-1)            ; e6 lda
      fcb (TY_IX1+2-1)            ; e7 ais
      fcb (TY_IX1+2-1)            ; e8 eor
      fcb (TY_IX1+2-1)            ; e9 adc
      fcb (TY_IX1+2-1)            ; ea ora
      fcb (TY_IX1+2-1)            ; eb add
      fcb (TY_IX1+2-1)            ; ec jmp
      fcb (TY_IX1+2-1)            ; ed jsr
      fcb (TY_IX1+2-1)            ; ee ldx
      fcb (TY_IX1+2-1)            ; ef stx
;
      fcb (TY_IX0+1-1)            ; f0 sub
      fcb (TY_IX0+1-1)            ; f1 cmp
      fcb (TY_IX0+1-1)            ; f2 sbc
      fcb (TY_IX0+1-1)            ; f3 cpx
      fcb (TY_IX0+1-1)            ; f4 and
      fcb (TY_IX0+1-1)            ; f5 bit
      fcb (TY_IX0+1-1)            ; f6 lda
      fcb (TY_IX0+1-1)            ; f7 ais
      fcb (TY_IX0+1-1)            ; f8 eor
      fcb (TY_IX0+1-1)            ; f9 adc
      fcb (TY_IX0+1-1)            ; fa ora
      fcb (TY_IX0+1-1)            ; fb add
      fcb (TY_IX0+1-1)            ; fc jmp
      fcb (TY_IX0+1-1)            ; fd jsr
      fcb (TY_IX0+1-1)            ; fe ldx
      fcb (TY_IX0+1-1)            ; ff stx
;
_ext_6_tbl:
      fcb (TY_SP1+3-1)            ; 60 neg
      fcb (TY_CBS+4-1)            ; 61 cbeq
      fcb (TY_INV+1-1)            ; 62 invalid
      fcb (TY_SP1+3-1)            ; 63 com
      fcb (TY_SP1+3-1)            ; 64 lsr
      fcb (TY_INV+1-1)            ; 65 invalid
      fcb (TY_SP1+3-1)            ; 66 ror
      fcb (TY_SP1+3-1)            ; 67 asr
      fcb (TY_SP1+3-1)            ; 68 lsl
      fcb (TY_SP1+3-1)            ; 69 rol
      fcb (TY_SP1+3-1)            ; 6a dec
      fcb (TY_CBS+4-1)            ; 6b dbnz
      fcb (TY_SP1+3-1)            ; 6c inc
      fcb (TY_SP1+3-1)            ; 6d tst
      fcb (TY_INV+1-1)            ; 6e invalid
      fcb (TY_SP1+3-1)            ; 6f clr
;
_ext_ef_tbl:
      fcb (TY_SP2+4-1)            ; 9e+d0 sub
      fcb (TY_SP2+4-1)            ; 9e+d1 cmp
      fcb (TY_SP2+4-1)            ; 9e+d2 sbc
      fcb (TY_SP2+4-1)            ; 9e+d3 cpx
      fcb (TY_SP2+4-1)            ; 9e+d4 and
      fcb (TY_SP2+4-1)            ; 9e+d5 bit
      fcb (TY_SP2+4-1)            ; 9e+d6 lda
      fcb (TY_SP2+4-1)            ; 9e+d7 ais
      fcb (TY_SP2+4-1)            ; 9e+d8 eor
      fcb (TY_SP2+4-1)            ; 9e+d9 adc
      fcb (TY_SP2+4-1)            ; 9e+da ora
      fcb (TY_SP2+4-1)            ; 9e+db add
      fcb (TY_INV+1-1)            ; 9e+dc Invalid
      fcb (TY_INV+1-1)            ; 9e+dd Invalid
      fcb (TY_SP2+4-1)            ; 9e+de ldx
      fcb (TY_SP2+4-1)            ; 9e+df stx
;
      fcb (TY_SP1+3-1)            ; 9e+e0 sub
      fcb (TY_SP1+3-1)            ; 9e+e1 cmp
      fcb (TY_SP1+3-1)            ; 9e+e2 sbc
      fcb (TY_SP1+3-1)            ; 9e+e3 cpx
      fcb (TY_SP1+3-1)            ; 9e+e4 and
      fcb (TY_SP1+3-1)            ; 9e+e5 bit
      fcb (TY_SP1+3-1)            ; 9e+e6 lda
      fcb (TY_SP1+3-1)            ; 9e+e7 ais
      fcb (TY_SP1+3-1)            ; 9e+e8 eor
      fcb (TY_SP1+3-1)            ; 9e+e9 adc
      fcb (TY_SP1+3-1)            ; 9e+ea ora
      fcb (TY_SP1+3-1)            ; 9e+eb add
      fcb (TY_INV+1-1)            ; 9e+ec Invalid
      fcb (TY_INV+1-1)            ; 9e+ed Invalid
      fcb (TY_SP1+3-1)            ; 9e+ee ldx
      fcb (TY_SP1+3-1)            ; 9e+ef stx
;
;     End if DISASM conditional
;
#endif
