$page

;*****************************************************************************
; SET_DDS - This routine sends the 5-byte DDS control word (DDS_w4 through
;           DDS_w0, LSB to MSB in each) to the DDS chip by serial data transfer.

Set_DDS:
         ldhx    #DDS_w4            ;point to the start of the DDS programming words
         mov     #$05,byte_count    ;we'll process 5 bytes

nextbyt: mov     X+,byte2send       ;get the DDS word (and increment pointer H:X)
         mov     #$08,bit_count     ;we'll process 8 bits

nextbit: ror     byte2send          ;test next LSB for 0 or 1
         bcc     _send0             ;send a 0 if carry clear

_send1:  bset    DDS_data,portf     ;else send a one
         bset    DDS_clk,portf      ;Toggle write clock
         bclr    DDS_clk,portf
         bra     _check

_send0:  bclr    DDS_data,portf     ;Send zero
         bset    DDS_clk,portf      ;Toggle write clock
         bclr    DDS_clk,portf

_check:  dbnz    bit_count,nextbit  ;get next DDS programming bit if not done
         dbnz    byte_count,nextbyt ;get next DDS programming byte if not done

         bset    DDS_load,portf     ;else done - send DDS load signal
         bclr    DDS_load,portf
         rts

;**********************************************************************************
; INIT_DDS - This routine puts the DDS into serial mode at power-on time

Init_DDS:
         lda     DDSchip            ;running AD9850 DDS chip?
         beq     idds_exit          ;yes, no special initialization needed
         cmp     #2
         beq     idds_exit          ;same too for AD9851 running at 1x refclk

              ;init the AD9851 DDS chip by turning on the 6x PLL

         bset    DDS_clk,portf      ;strobe Clock to get hardware bits D0, D1 and D2 into the input reg
         bclr    DDS_clk,portf
         bset    DDS_load,portf     ;and strobe Load to get them into the control reg
         bclr    DDS_load,portf
         jsr     wr_zeros_ad9851    ;write 32 zeros to AD9851 with 6x PLL enabled
         jsr     wr_zeros_ad9851    ;write 32 zeros to AD9851 with 6x PLL enabled

              ;and exit
idds_exit:
          rts

              ;write 32 zeros to AD9851 with 6x PLL enabled
wr_zeros_ad9851:
         clr     DDS_w4
         clr     DDS_w3
         clr     DDS_w2
         clr     DDS_w1
         mov     #$01,DDS_w0        ; and 6x multiplier ...
         jsr     set_DDS            ; to flush residual data in DDS core logic
         rts

$page
;**************************************************************************
;*******************      EEPROM Driver      ******************************
;**************************************************************************
;
;--------------------------------------------------------------------------
; Write_EEPROM_Byte -- The target EEPROM address H:X (in the range 800-9FF)
;                      is first erased, then written with the byte supplied
;                      in reg A.

Write_EEPROM_Byte:
        psha                            ;save the data to be written
        jsr     Erase_EEPROM_Byte       ;erase the target location
        lda     #(EERAS1|EERAS0)
        coma                            ;clear EERAS1 & EERAS0 in EECR
        and     EECR
        sta     EECR
        ora     #EELAT                  ;set EELAT in EECR
        sta     EECR
        pula                            ;and retrieve the data
        sta     ,x                      ;Write data to the desired EEPROM address
        lda     #EEPGM                  ;Set the EEPGM bit in the EECR
        ora     EECR
        sta     EECR
        jsr     Wait100ms                ;Wait 10ms to program the byte.
        lda     #EEPGM                  ;clear EEPGM
        coma
        and     EECR
        sta     EECR
        jsr     Wait100us               ;Wait 100us for pgm voltage to fall.
        lda     #EELAT
        coma                            ;clear EELAT
        and     EECR
        sta     EECR
        rts

;--------------------------------------------------------------------------
; Erase_EEPROM_All -- Bulk erase all EEPROM locations (800-9FF)

Erase_EEPROM_all:
        ldhx    #$800                   ;any location in EEPROM area will do bulk erase
        lda     EECR
        ora     #(EERAS1|EERAS0)
        sta     EECR
        bra     ee_1

;--------------------------------------------------------------------------
; Erase_EEPROM_Byte -- Erase target EEPROM address H:X (in the range 800-9FF)

Erase_EEPROM_byte:
        lda     EECR
        and     #$EF                    ;clear EERAS1 (b4)
        ora     #$08                    ;set EERAS0 (b3)
        sta     EECR
ee_1:   ora     #EELAT                  ;set EELAT in EECR
        sta     EECR
        sta     ,x                      ;write any data to the target location erases
        lda     #EEPGM                  ;Set the EEPGM bit in the EECR
        ora     EECR
        sta     EECR
        jsr     Wait100ms                ;Wait 10ms for byte erase
        lda     #EEPGM
        coma                            ;clear EEPGM
        and     EECR
        sta     EECR
        jsr     Wait100us               ;Wait 100us for pgm voltage to fall.
        lda     #EELAT
        coma                            ;clear EELAT
        and     EECR
        sta     EECR
        rts

;--------------------------------------------------------------------------
; Init_EEPROM -- Initialize the EEPROM configuration registers
;                This is a nonvolatile register setting

Init_EEPROM:
        jsr     EEPROM_On               ;Turn on array for operation

        lda     #$10                    ;lower 4 bits are positions for EEPROM bank enables
        ldhx    #EENVR
        jsr     Write_EEPROM_Byte       ;enable the EEPROM
        lda    ,x                       ;puts new settings into effect

        lda     #$80                    ;set upper bit EEDIVSECD to allow the values to be modified later
        ldhx    #EEDIVHNVR
        jsr     Write_EEPROM_Byte       ;write the high divider byte
        lda     #172                    ;divider value is 170 (based on CGMXCLK clock of 4.9152 MHz)
        ldhx    #EEDIVLNVR
        jsr     Write_EEPROM_Byte       ;write the low divider byte

        rts

;--------------------------------------------------------------------------
; EEPROM_On -- Turns on the EEPROM array for operation (default = ON)

EEPROM_On:
        lda     #EEOFF
        coma
        and     EECR                    ;clear EEOF in EECR
        sta     EECR
        rts

;--------------------------------------------------------------------------
; EEPROM_Off -- Turns off the EEPROM array.  Provides for lower power
;               operation (default = ON)

EEPROM_Off:
        lda     EECR
        ora     #EEOFF                  ;set EEOFF on EECR
        sta     EECR
        rts

;--------------------------------------------------------------------------
; EEPROM_Security_On -- Turns on the EEPROM array security bit
;                       This is a nonvolatile setting. (Initially = disabled)
;                       Takes effect after reset.

EEPROM_Sec_On:
        mov    EEPRTCT,temp             ;set up temp to hold inverse of EEPRTCT bit
        com    temp                     ; to enable clearing of bit in EENVR below
        ldhx   #EENVR
        lda    ,x
        and    temp                     ;0=secure
        jsr    Write_EEPROM_Byte
        rts

;--------------------------------------------------------------------------
; EEPROM_Security_Off -- Turns off the EEPROM array security bit
;                        This is a nonvolatile setting. (Initially = disabled)
;                        Changes take effect after reset.

EEPROM_Sec_Off:
        ldhx   #EENVR
        lda    ,x
        ora    #EEPRTCT                 ;1=security disabled
        jsr    Write_EEPROM_Byte
        lda    ,x                       ;puts new settings into effect
        rts

;--------------------------------------------------------------------------
; EEPROM_Disable -- Protects the selected EEPROM block (0,1,2,3)
;                   from being written or erased.  Block indicated
;                   in corresponding bits of reg A.
;                   All four blocks are initially unprotected. (3:0 =0)
;                   This is a nonvolile setting. (Initially = disabled)
;                   Changes take effect after a reset.

EEPROM_Disable:
        and    #$0F                     ;isolate the block bits (3:0)
        sta    temp                     ;temp save selected block(s)
        ldhx   #EENVR
        lda    ,x
        ora    temp                     ;disable (protect) the selected bits/banks
        jsr    Write_EEPROM_Byte
        lda    ,x                       ;puts new seetings into effect
        rts

;--------------------------------------------------------------------------
; EEPROM_Enable -- Enables the selected EEPROM block (0,1,2,3)
;                  to be being written or erased.  Block indicated
;                  in corresponding bits of reg A.
;                  All four blocks are initially unprotected. (3:0 =0)
;                  This is a nonvolatile setting. (Initially = disabled)
;                  Changes take effect after a reset.

EEPROM_Enable:
        and    #$0F                     ;isolate the block bits (3:0)
        coma                            ;invert (0 means enable)
        sta    temp                     ;temp save selected block(s)
        ldhx   #EENVR
        lda    ,x
        and    temp                     ;enable (unprotect) the selected bits/banks
        jsr    Write_EEPROM_Byte
        lda    ,x                       ;puts new seetings into effect
        rts
