$BASE 10T
;**************************************************************************
;                            "HelloDDS"
;                       DIGITAL QRP BREADBOARD
;
;                        George Heron, N2APB
;                        2419 Feather Mae Ct.
;                        Forest Hill, MD  21050
;                        email: n2apb@amsat.org
;
; "HelloDDS" is a command line driven controller for an AD9850 DDS
; chip connected in a simple serial load fashion to the HC908 Daughtercard.
; Upon program initiation, the application offers 5 choices to the operator over
; the serial console port.
;
; The operator may ENTER a give FREQUENCY to the DDS by selecting Option 1. In this
; case the DDS is commanded to immediately generate the specified Frequency.
;
; Alternatively, the operator may set up the START, END and STEP SIZE parameters
; for a FREQUENCY SWEEP by entering Options 2, 3, 4 and 6, respectively.
;
; At any time the operator may SHOW the SETTINGS for all the frequency/scan parameters
; by entering Option 5.
;
; This is still a simple program and not much error checking is performed.  The
; operator is cautioned to always enter 7 digits for frequency parameter entry.
; Frequencies, represented by XXXXXXX, have a valid range from 0000000 to 9999999.
; However note that the DDS only is able to reliably generate frequencies
; up to 30 MHz, or 3000000.
;
; As can be seen when displayed to the serial port concole, the Frequency is represented
; as megahertz values in the form XX.XXXXX.
;
; Remember, do not enter a decimal point ... just always enter 7 digits.  (You'll get used
; to it, trust me!)
;
; HelloDDS sits in "User Application" area of memory ($8000-$EEFF) and co-resides
; with the HCmon debug monitor also on the Daughtercard (at $EF00-$FFDE).
;
; For complete documentation and design details, see the web pages devoted for this
; project at www.njqrp.org/hc908_resource_page
;
; This code is intended to run on a Motorola 68HC908AB32 microcontroller and is
; best assembled within the Integrated Development Environment editor, provided free
; by P&E Micro as part of their HC908 development tool suite.  See the HC908 Resource Page
; for full details on obtaining this development software.
;
; Revision History:
; ----------------
; 1.0      04/22/2003      Initial release of sample shell
; 1.2      05/27/2003      Added details to make it actually scan
; 2.0      01/12/2007      Added support for DDS-60 card
;
;**************************************************************************
; Copyright 2003-2007 by G. Heron, N2APB.  All rights reserved. For personal, non-profit use only.
;**************************************************************************

        include "AB_REGS.asm"            ;MC68HC908GP32 register defimitions
        include "macros.asm"             ;Macro instruction definition
        include "local.asm"              ;Local system variables

RAMStart     equ  $0053         ;User App RAM is from 53 to 3FF. (HCmon RAM from 400-44F)
EEPROM       equ  $800          ;EEPROM mem is 800-9FF (512 bytes)
RomStart     equ  $8000         ;Flash mem is 8000 to FFFF (32 MB)

                                ;PortF pin usage
Mon_Jumper   equ  7               ;Monitor jumper
HeartbeatLED equ  6               ;Heartbeat LED
DDS_load     equ  2               ;DDS load pin
DDS_clk      equ  1               ;DDS clock pin
DDS_data     equ  0               ;DDS data pin


    org RamStart

_hbufpos     rmb  2               ;buffer position temp storage

Timeout1     ds   1               ;Allows three timeout routines to be called each of which
Timeout2     ds   1               ;can run for up to ~ 1/2 second.
Timeout3     ds   1

EEbyte_count ds   1               ;EEPROM byte counter
EEline_count ds   1               ;EEPROM line counter
temp         ds   1               ;temp used in EEPROM driver
to_addr      ds   2               ;index pointer storage locs
from_addr    ds   2

freq         ds   4               ;frequency represented as 4-byte binary
ad9850       ds   4               ;temp working register
multcount    ds   1               ;32x32 multiplier counter
temp1        ds   1
temp2        ds   1
temp3        ds   1

bit_count    ds   1               ;DDS bit counter
byte_count   ds   1               ;DDS byte counter
byte2send    ds   1               ;DDS byte to send
DDS_w4       ds   1               ;DDS programming register W4 (freq-b7  ...  freq-b0)
DDS_w3       ds   1               ;DDS programming register W3 (freq-b15 ...  freq-b8)
DDS_w2       ds   1               ;DDS programming register W2 (freq-b23 ...  freq-b16)
DDS_w1       ds   1               ;DDS programming register W1 (freq-b31 ...  freq-b24)
DDS_w0       ds   1               ;DDS programming register W0 (phase-b4 ...  control-0)


Frequency    ds   10              ;frequency string
Start_Freq   ds   10              ;start_freq string
End_Freq     ds   10              ;end_freq string
Step_Size    ds   10              ;freq step size

                                  ;7 BCD data (binary coded decimal) digits
                                  ;  for frequency - MSD (1MHz) to LSD (10 Hz)
BCD_dat      ds   7               ;BCD_dat+0 = 10,000,000 Hz BCD digit position
                                  ;BCD_dat+1 =  1,000,000 Hz BCD digit position
                                  ;BCD_dat+2 =    100,000 Hz BCD digit position
                                  ;BCD_dat+3 =     10,000 Hz BCD digit position
                                  ;BCD_dat+4 =      1,000 Hz BCD digit position
                                  ;BCD_dat+5 =        100 Hz BCD digit position
                                  ;BCD_dat+6 =         10 Hz BCD digit position

Start_BCD    ds   7               ;start freq BCD registers
End_BCD      ds   7               ;end freq BCD registers
Step_BCD     ds   7               ;step BCD registers

op1          ds   4               ;operand 1 for add & multiply routine
op2          ds   4               ;operand 2
DDSchip      ds   1               ;AD9850=0, AD9851=1
count1       ds   1
count2       ds   1
start_index  ds   2

dest         ds   2               ;destination address store for move routines
source       ds   2               ;source address store for move routines
move_cntr    ds   2               ;counter for Save_Input routine

count_lo     ds   1               ;low byte counter for timer routines
count_hi     ds   1               ;high byte counter for timer routines

msg_cntr     ds   1               ;counter for number of menu messages in list
msg_pointer  ds   2               ;storage for message pointer
send_cntr    ds   1               ;counter for number of bytes to send in Send_Command


    org $040A
_inbuf       rmb  RXBLEN          ;common buffer with HCmon

    org RomStart

;**************************************************************************
;   Vector Jump Table (located at start of User Area = 8000)
;**************************************************************************

_user_adc         jmp    Dummy_ISR      ; Insert jump to ADC vector code
_user_keyboard    jmp    Dummy_ISR      ; Insert jump to Keyboard vector code
_user_scitx       jmp    Dummy_ISR      ; Insert jump to SCI transmit vector code
_user_scirx       jmp    Dummy_ISR      ; Insert jump to SCI receive vector code
_user_scierr      jmp    Dummy_ISR      ; Insert jump to SCI error vector code
_user_timbch3     jmp    Dummy_ISR      ; Insert jump to TIMB Channel 3 Vector code
_user_timbch2     jmp    Dummy_ISR      ; Insert jump to TIMB Channel 2 Vector code
_user_spitx       jmp    Dummy_ISR      ; Insert jump to SPI transmit vector code
_user_spirx       jmp    Dummy_ISR      ; Insert jump to SPI receive vector code
_user_timb_of     jmp    Dummy_ISR      ; Insert jump to TIMB Overflow Vector code
_user_timb_ch1    jmp    Dummy_ISR      ; Insert jump to TIMB Channel 1 Vector code
_user_timb_ch0    jmp    Dummy_ISR      ; Insert jump to TIMB Channel 0 Vector code
_user_tima_of     jmp    Dummy_ISR      ; Insert jump to TIMA Overflow Vector code
_user_tima_ch3    jmp    Dummy_ISR      ; Insert jump to TIMA Channel 3 Vector code
_user_tima_ch2    jmp    Dummy_ISR      ; Insert jump to TIMA Channel 2 Vector code
_user_tima_ch1    jmp    T_ISR_A1       ; Insert jump to TIMA Channel 1 Vector code
_user_tima_ch0    jmp    T_ISR_A0       ; Insert jump to TIMA Channel 0 Vector code
_user_tim_of      jmp    Dummy_ISR      ; Insert jump to TIM Overflow Vector code
_user_pll         jmp    Dummy_ISR      ; Insert jump to PLL Vector code
_user_irq         jmp    Dummy_ISR      ; Insert jump to ~IRQ1 Vector code

_HCmon_LCD_disp   jmp    HCmon_LCD_disp ; vector (at $803C) for HCmon
                                        ; to display message to LCD

;**************************************************************************
;**********************      USER APP ENTRY POINT     *********************
;**************************************************************************
;        This is the point ($8040) where User Code starts executing       *
;        after coming from the Monitor "X" command (eXecute User App)     *
;        or after board gets a RESET (when Monitor jumper is off)         *
;**************************************************************************

        org     $8040

Start:
        sei                            ;disable all interrupts
        ldhx    #$01FF                 ;initialize
        txs                            ;  the stack pointer (why not 3FF?)

;**************************************************************************
;*****************      USER APPPLICATION MAINLINE      *******************
;**************************************************************************

User_Main:
        mov     #3,timeout1             ;Start an A/D conversion in 3*2ms=6ms
        lda     #100
        jsr     delay
        jsr     Init_All                ;initialize the system

Main_Loop:
	ldhx 	#DDSmenu_msg
	jsr 	_puts	                ;print the Banner message
	ldhx	#Command_msg	        ;point to the 'Command' string
	jsr	_puts	                ;and print it to the terminal
	jsr	_gets	                ;get an input command
	lda	_inbuf	                ;get the entered character
	cmp	#'1'	                ;compare the input to '1'
        bne     m2                      ;  not a '1', go check for a '2'
	jsr	enter_freq	        ;  if '1', call the Enter Freq routine
        bra     m7                      ;  and continue
m2:	cmp	#'2'	                ;compare the input to '2'
        bne     m3                      ;  not a '1', go check for a '3'
	jsr	set_start_freq	        ;  if '2', call the Set Start Freq routine
        bra     m7                      ;  and continue
m3:	cmp	#'3'	                ;compare the input to '3'
        bne     m4                      ;  not a '3', go check for a '4'
	jsr	set_end_freq	        ;  if '3', call the Set End Freq routine
        bra     m7                      ;  and continue
m4:	cmp	#'4'	                ;compare the input to '4'
        bne     m5                      ;  not a '4', go check for a '5'
	jsr	set_step_size	        ;  if '4', call the Set Step Size routine
        bra     m7                      ;  and continue
m5:     cmp     #'5'                    ;compare the input to '5'
        bne     m6                      ;  not a '5', go check for a '6'
        jsr     show_settings           ;  if '5', call the show settings routine
        bra     m7                      ;  and continue
m6:	cmp	#'6'	                ;compare the input to '6'
        bne     m7                      ;  not a '5', entry unrecognized.
	jsr	sweep_dds	        ;  if '5', call the Sweep DDS routine
m7:	cmp	#'.'	                ;compare the input to '.' DEBUG MONITOR ENTRY
        bne     m8                      ;  not a '.', entry unrecognized.
	swi                             ;  if '.', enter HCMON
m8:     jsr     _crlf                   ;unrecognized input, no nothing
	bra	Main_Loop	        ;and go get another command

DDSmenu_msg:
	fcb 	CR,LF,LF,'HELLO DDS v2.0 (DDS30)',CR,LF
	fcb 	'1) Enter freq',CR,LF
	fcb 	'2) Set start freq',CR,LF
	fcb	'3) Set end freq',CR,LF
	fcb	'4) Set step size',CR,LF
        fcb     '5) Show Settings',CR,LF
	fcb	'6) Sweep DDS',CR,LF,0

Command_msg:
	fcb 	CR,LF,'Command: ',0

;********************************************************************
;*****************        COMMAND PROCESSORS        *****************
;********************************************************************

;------------------------
; 1) Enter Frequency
;------------------------

Enter_Freq:
	ldhx	#EnterFreq_msg	        ;point to the 'Enter Freq' string
	jsr	_puts	                ;and print it to the terminal
	jsr	_gets	                ;get a string of numbers
        ldhx    #frequency
        jsr     Save_Input              ;save input string at Frequency
        ldhx    #frequency
        jsr     ConvAsc2Bcd             ;convert the 7-digit ascii string at Frequency
                                        ; to 7 BCD values in BCD_dat 0:6
        jsr     print_BCD_Dat           ;display the frequency
        jsr     Mult                    ;Multiply 7 Frequency BCDs by weighted digit values to get DDS word
        jsr	Set_DDS	                ;set the DDS output to that freq
        jsr     wait10ms
        jsr	Set_DDS	                ;set the DDS output to that freq
        jsr     wait10ms
        jsr	Set_DDS	                ;set the DDS output to that freq
        jsr     wait10ms
        rts                             ;and go get next command

EnterFreq_msg:
	fcb	CR,LF,'Enter Freq: ',0

;------------------------
; 2) Set Start Frequency
;------------------------

Set_Start_Freq:
	ldhx	#Start_msg	        ;point to the 'Start Freq' string
	jsr	_puts	                ;and print it to the terminal
	jsr	_gets	                ;get a string of numbers
        ldhx    #start_freq
        jsr     Save_Input              ;save input string at Start_Freq
        ldhx    #start_freq
        jsr     ConvAsc2Bcd             ;convert 7-digit ascii string at Start_Freq
                                        ; to 7 BCD values in BCD_dat 0:6
        ldhx    #start_BCD
        jsr     Save_BCD                ;save BCD data to 7 start_BCD locations
        rts                             ;and go get next command

Start_msg:
        fcb     CR,LF,'Enter Start Freq: ',0

;------------------------
; 3) Set End Frequency
;------------------------

Set_End_Freq:
	ldhx	#End_msg	        ;point to the 'End Freq' string
	jsr	_puts	                ;and print it to the terminal
	jsr	_gets	                ;get a string of numbers
        ldhx    #end_freq
        jsr     Save_Input              ;save input string at End Freq
        ldhx    #end_freq
        jsr     ConvAsc2Bcd             ;convert 7-digit ascii string at End_Freq
                                        ; to 7 BCD values in BCD_dat 0:6
        ldhx    #end_BCD
        jsr     Save_BCD                ;save BCD data to 7 end_BCD locations
        rts                             ;and go get next command

End_msg:
        fcb     CR,LF,'Enter End Freq: ',0

;------------------------
; 4) Set Step Size
;------------------------

Set_Step_Size:
	ldhx	#Step_msg	        ;point to the 'Set Step Size' string
	jsr	_puts	                ;and print it to the terminal
	jsr	_gets	                ;get a string of numbers
        ldhx    #step_size
        jsr     Save_Input              ;save input string at Step_Size
        ldhx    #Step_size
        jsr     ConvAsc2Bcd             ;convert 7-digit ascii string at Step_Size
                                        ; to 7 BCD values in BCD_dat 0:6
        ldhx    #step_BCD
        jsr     Save_BCD                ;save BCD data to 7 step_BCD locations
        rts                             ;and go get next command

Step_msg:
        fcb     CR,LF,'Enter Step Size: ',0

;------------------------
; 5) Show Current Settings
;------------------------

Show_Settings:
        ldhx    #Current_msg
        jsr     _puts                   ;send "Current Settings..."

        ldhx    #f_msg
        jsr     _puts                   ;send "Frequency = "
        ldhx    #frequency
        jsr     _puts                   ;send the frequency string
        jsr     _crlf

        ldhx    #startmsg
        jsr     _puts                   ;send "Start = "
        ldhx    #start_freq
        jsr     _puts                   ;send the start frequency
        jsr     _crlf

        ldhx    #endmsg
        jsr     _puts                   ;send "End = "
        ldhx    #end_freq
        jsr     _puts                   ;send the end frequency
        jsr     _crlf

        ldhx    #stepmsg
        jsr     _puts                   ;send "Step size = "
        ldhx    #step_size
        jsr     _puts                   ;send the step size
        jsr     _crlf

        rts

Current_msg:    fcb     CR,LF,'Current Settings:',CR,LF,0
f_msg:          fcb     '  Frequency: ',0
startmsg:       fcb     '      Start: ',0
endmsg:         fcb     '        End: ',0
stepmsg:        fcb     '       Step: ',0

;------------------------
; 6) Sweep DDS
;------------------------

Sweep_DDS:
	ldhx	#Sweep_msg	        ;point to the 'Sweeping' string
	jsr	_puts	                ;and print it to the terminal

        mov     start_BCD+0,BCD_Dat+0   ;move 7 digit Start_BCD ==> BCD_Dat
        mov     start_BCD+1,BCD_Dat+1
        mov     start_BCD+2,BCD_Dat+2
        mov     start_BCD+3,BCD_Dat+3
        mov     start_BCD+4,BCD_Dat+4
        mov     start_BCD+5,BCD_Dat+5
        mov     start_BCD+6,BCD_Dat+6

sweep1: jsr     Add_Step_Size           ;add the step size to the Freq_BCD registers
        jsr     Mult                    ;Multiply 7 Frequency BCDs by weighted digit values to get DDS word
        jsr	Set_DDS	                ;set the DDS output to that freq
        jsr     wait1ms
        jsr     Check_For_End           ;compare Freq_BCD regs to see if it's past the End_BCD reg value
        bcc     sweep1                  ;not done yet
        rts                             ;and go get next command

Sweep_msg:
        fcb     CR,LF,'Sweeping DDS ...',0


;*********************************************************************************
; Init_All
;*********************************************************************************

Init_all:
             ;configure the ports
        mov     #$47,ddrf              ;set port F6, F2-F0 as outputs

             ;initialize the output ports
        bclr    DDS_load,portf
        bclr    DDS_clk,portf
        bclr    DDS_data,portf

        sta	copctl	                ;clear the COP counter

        clr     frequency              ;initialize string (to null)
        clr     start_freq             ;initialize string (to null)
        clr     end_freq               ;initialize string (to null)
        clr     step_size              ;initialize string (to null)

        mov     #'1',Frequency+0       ;init the Frequency registers to 10.000.00 MHz
        mov     #'0',Frequency+1
        mov     #'0',Frequency+2
        mov     #'0',Frequency+3
        mov     #'0',Frequency+4
        mov     #'0',Frequency+5
        mov     #'0',Frequency+6
        clr     Frequency+7            ;terminator

        mov     #$01,BCD_Dat+0         ;Init the frequency BCD_Dat registers to 10.000.00 MHz
        mov     #$00,BCD_Dat+1
        mov     #$00,BCD_Dat+2
        mov     #$00,BCD_Dat+3
        mov     #$00,BCD_Dat+4
        mov     #$00,BCD_Dat+5
        mov     #$00,BCD_Dat+6

        mov     #$00,Start_BCD+0        ;Init the Start_BCD registers to 01.000.00 MHz
        mov     #$01,Start_BCD+1
        mov     #$00,Start_BCD+2
        mov     #$00,Start_BCD+3
        mov     #$00,Start_BCD+4
        mov     #$00,Start_BCD+5
        mov     #$00,Start_BCD+6

        mov     #'0',Start_Freq+0       ;init the Start_Freq registers to 01.000.00 MHz
        mov     #'1',Start_Freq+1
        mov     #'0',Start_Freq+2
        mov     #'0',Start_Freq+3
        mov     #'0',Start_Freq+4
        mov     #'0',Start_Freq+5
        mov     #'0',Start_Freq+6
        clr     Start_Freq+7            ;terminator

        mov     #$01,End_BCD+0          ;Init the End_BCD registers to 10.000.00 MHz
        mov     #$00,End_BCD+1
        mov     #$00,End_BCD+2
        mov     #$00,End_BCD+3
        mov     #$00,End_BCD+4
        mov     #$00,End_BCD+5
        mov     #$00,End_BCD+6

        mov     #'1',End_Freq+0         ;init the End_Freq registers to 10.000.00 MHz
        mov     #'0',End_Freq+1
        mov     #'0',End_Freq+2
        mov     #'0',End_Freq+3
        mov     #'0',End_Freq+4
        mov     #'0',End_Freq+5
        mov     #'0',End_Freq+6
        clr     End_Freq+7              ;terminator

        mov     #$00,Step_BCD+0         ;init the Step BCD registers to 00.010.00 MHz
        mov     #$00,Step_BCD+1
        mov     #$00,Step_BCD+2
        mov     #$01,Step_BCD+3
        mov     #$00,Step_BCD+4
        mov     #$00,Step_BCD+5
        mov     #$00,Step_BCD+6

        mov     #'0',Step_Size+0        ;init the Step_Freq registers to 00.010.00 MHz
        mov     #'0',Step_Size+1
        mov     #'0',Step_Size+2
        mov     #'1',Step_Size+3
        mov     #'0',Step_Size+4
        mov     #'0',Step_Size+5
        mov     #'0',Step_Size+6
        clr     Step_Size+7             ;terminator

        jsr     Init_Timer
        jsr     Init_EEPROM

        mov     #0,DDSchip              ;init system to DDS-30 operation
        jsr     Init_DDS

        clr     timeout1                ;init timeouts (0=off)
        clr     timeout2
        clr     timeout3

        cli                             ;allow interrupts to happen

        rts

;**************************************************************************
;   Save_Input - takes user input string from _inbuf and places it at H:X
;**************************************************************************

Save_Input:
        sthx    dest                    ;save destinate addr
        ldhx    #0
        sthx    move_cntr               ;init the move counter
si_1:   ldhx    move_cntr               ;get the counter
        cphx    #8                      ;max digit entry currently at 8
        beq     si_2                    ;only move 8 digits, even if terminator not yet encountered
        lda     _inbuf,x                ;use counter as offset for getting char from _inbuf string
        aix     #1                      ;increment the counter
        sthx    move_cntr               ;and save it
        ldhx    dest                    ;get the destination addr
        sta     ,x                      ;store the input character there
        beq     si_2                    ;if it's 0, it's the end
        aix     #1                      ;else bump the dest addr
        sthx    dest                    ;and save it away
        bra     si_1                    ;do again, until zero end-of-string found
si_2:
        rts

;**************************************************************************
;   Save_BCD - Save the 7 BCD digits from BCD_dat 0:6 to location pointer H:X
;**************************************************************************

Save_BCD:
        sthx    dest                    ;save the destination address
        ldhx    #0                      ;init counter
        sthx    move_cntr               ;  used to count ascii digits converted to BCD_dat regs
sb1:    ldhx    move_cntr
        lda     BCD_dat,x               ;get the BCD data
        aix     #1                      ;bump the counter/pointer
        sthx    move_cntr
        ldhx    dest                    ;else, proceed to store BCD data into destination
        sta     ,x                      ;save the digit at destination addr
        aix     #1                      ;bump destination addr
        sthx    dest
        ldhx    move_cntr
        cphx    #7                      ;done with all 7 digits?
        bne     sb1                     ;not yet
        rts

;**************************************************************************
;   ConvASC2BCD -- Convert the ascii string at H:X to 7 BCD_dat registers
;              Assume 7-digit hertz value entered, 0000000 through 3000000.
;              Example: 0704100
;              (We'll get fancier later on to accommodate more digits and
;              eliminate leading zeros.)
;**************************************************************************

ConvAsc2Bcd:
        sthx    source                  ;save the source address
        ldhx    #0                      ;init counter
        sthx    move_cntr               ;  used to count ascii digits converted to BCD_dat regs
con1:   ldhx    source
        lda     ,x                      ;get digit
        beq     con2                    ;if 0 then it's the end of string
        jsr     _a2nibhex               ;convert ascii to hex nibble
        aix     #1                      ;bump source addr
        sthx    source
        ldhx    move_cntr
        sta     BCD_dat,x               ;store nibble in BCD_dat register
        aix     #1                      ;bump the move_cntr
        sthx    move_cntr
        cphx    #7                      ;done with all 7 digits?
        bne     con1                    ;no, continue with next digit
con2:   rts

;**************************************************************************************
; Add_Step_Size -- 7 BCD digits add operation: Step_BCD + Freq_BCD = Freq_BCD

Add_Step_Size:
        ldhx    #BCD_Dat+6
        lda     Step_BCD+6              ;get LS digit of Step_BCD
        jsr     add_BCD_digit           ;add the two BCD digits
        bcc     ass1                    ;if no carry, continue
        inc     BCD_Dat+5               ;increment due to CY
ass1:   aix     #-1                     ;point to next higher Freq_BCD position
        lda     Step_BCD+5              ;get LS digit of Step_BCD
        jsr     add_BCD_digit           ;add the two BCD digits
        bcc     ass2                    ;if no carry, continue
        inc     BCD_Dat+4               ;increment due to CY
ass2:   aix     #-1                     ;point to next higher Freq_BCD position
        lda     Step_BCD+4              ;get LS digit of Step_BCD
        jsr     add_BCD_digit           ;add the two BCD digits
        bcc     ass3                    ;if no carry, continue
        inc     BCD_Dat+3               ;increment due to CY
ass3:   aix     #-1                     ;point to next higher Freq_BCD position
        lda     Step_BCD+3              ;get LS digit of Step_BCD
        jsr     add_BCD_digit           ;add the two BCD digits
        bcc     ass4                    ;if no carry, continue
        inc     BCD_Dat+2               ;increment due to CY
ass4:   aix     #-1                     ;point to next higher Freq_BCD position
        lda     Step_BCD+2              ;get LS digit of Step_BCD
        jsr     add_BCD_digit           ;add the two BCD digits
        bcc     ass5                    ;if no carry, continue
        inc     BCD_Dat+1               ;increment due to CY
ass5:   aix     #-1                     ;point to next higher Freq_BCD position
        lda     Step_BCD+1              ;get LS digit of Step_BCD
        jsr     add_BCD_digit           ;add the two BCD digits
        bcc     ass6                    ;if no carry, continue
        inc     BCD_Dat+0               ;increment due to CY
ass6:   aix     #-1                     ;point to next higher Freq_BCD position
        lda     Step_BCD+0              ;get LS digit of Step_BCD
        jsr     add_BCD_digit           ;add the two BCD digits
ass7:   rts

;-------------
add_BCD_digit:
        clc
        add     ,x                      ;add LS digit of Freq_BCD
        cmp     #$0A                    ;result greater than 9?
        bcs     ab1                     ;no, continue
        add     #6                      ;yes, decimal adjust the result
        and     #$0F                    ;keep result to lower nibble
        sec                             ;set C to cause increment of next digit
        sta     ,x                      ;store result in Freq_BCD+n
        rts                             ;return with C indicating a carry occurred

ab1:    sta     ,x                      ;store result in Freq_BCD+n
        clc                             ;clear C to indicate no increment of next digit
        rts


;**************************************************************************************
; Check_For_End -- checks Frequency "BCD_Dat" regs to see if they are past the End_BCD regs

Check_For_End:
        ldhx    #BCD_Dat
        lda     End_BCD+0
        cmp     ,x                      ;compare 10 MHz digit
        beq     cf1                     ;if both zero or both the same, check next
        bcs     cf7                     ;exit if Freq beyond End
        jmp     cf8                     ;else exit with C=0 if not beyond End

cf1:    aix     #1
        lda     End_BCD+1
        cmp     ,x                      ;compare 1 MHz digit
        beq     cf2                     ;if both zero or both the same, check next
        bcs     cf7                     ;exit if Freq beyond End
        jmp     cf8                     ;else exit with C=0 if not beyond End

cf2:    aix     #1
        lda     End_BCD+2
        cmp     ,x                      ;compare 100 kHz digit
        beq     cf3                     ;if both zero or both the same, check next
        bcs     cf7                     ;exit if Freq beyond End
        jmp     cf8                     ;else exit with C=0 if not beyond End

cf3:    aix     #1
        lda     End_BCD+3
        cmp     ,x                      ;compare 10 kHz digit
        beq     cf4                     ;if both zero or both the same, check next
        bcs     cf7                     ;exit if Freq beyond End
        jmp     cf8                     ;else exit with C=0 if not beyond End

cf4:    aix     #1
        lda     End_BCD+4
        cmp     ,x                      ;compare 1 kHz digit
        beq     cf5                     ;if both zero or both the same, check next
        bcs     cf7                     ;exit if Freq beyond End
        jmp     cf8                     ;else exit with C=0 if not beyond End

cf5:    aix     #1
        lda     End_BCD+5
        cmp     ,x                      ;compare 100 Hz digit
        beq     cf6                     ;if both zero or both the same, check next
        bcs     cf7                     ;exit if Freq beyond End
        jmp     cf8                     ;else exit with C=0 if not beyond End

cf6:    aix     #1
        lda     End_BCD+6
        cmp     ,x                      ;compare 10 Hz digit
        beq     cf8                     ;if both zero or both the same, just exit with c=0
        bcs     cf7                     ;exit with C=1 if Freq beyond End
        jmp     cf8                     ;failsafe

            ;"end reached" - exit with C=1
cf7:    rts

            ;"end not reached" - exit with c=0
cf8:    clc
        rts

;**************************************************************************************
; Print_Freq_BCD -- Print the current Freqency as represented in the 7 BCD digits at BCD_Dat

print_BCD_Dat:
        jsr     _crlf
        jsr     _space
        jsr     _space
        lda     BCD_Dat+0
        jsr     _pr_nibble
        lda     BCD_Dat+1
        jsr     _pr_nibble
        lda     #'.'
        jsr     _putch
        lda     BCD_Dat+2
        jsr     _pr_nibble
        lda     BCD_Dat+3
        jsr     _pr_nibble
        lda     BCD_Dat+4
        jsr     _pr_nibble
        lda     BCD_Dat+5
        jsr     _pr_nibble
        lda     BCD_Dat+6
        jsr     _pr_nibble
        jsr     _space
        ldhx    #mHz_msg
        jsr     _puts
        rts

mHz_msg:  fcb   'MHz',0

;*****************************************************************************
; MULT -- 7 Digit BCD by 32 bit binary mulitply
; The 7 BCD digits stored in BCD_dat are each multiplied by a precalculated digit
; weight. The sum of all 7 digits times their weight is stored in the DDS_wN registers.
; (Weights of each digit are determined by using Analog Devices "Interactive Design Tool"
;  for either the AD9850 or the AD9851.)

Mult:   clr         temp1
        clr         DDS_w4
        clr         DDS_w3
        clr         DDS_w2
        clr         DDS_w1
        clr         DDS_w0             ;use this line for the AD9850 with 100 MHz refclk
        lda         DDSchip            ;using AD9850 DDS chip?
        beq         m10                ;yes, just continue (with DDS_w0 = 00)
        cmp         #2                 ;using AD9851 w/ 1x refclk?
        beq         m10                ;yes, just continue (with DDS_w0 = 00)
        lda         #$01               ;else, use 01 for the AD9851 with 6x refclk multiplier
        sta         DDS_w0

m10:    mov         #$AD,OP1+3
        mov         #$01,OP1+2
        lda         DDSchip            ;which DDS chip?
        beq         m10a               ;yes, using the AD9850 so just continue
        cmp         #1                 ;AD9851 w/ 6x PLL?
        beq         m10b
        mov         #$98,OP1+3     ;no, using the AD9851 at 1x refclk so load these constants instead
        mov         #$05,OP1+2
        bra         m10a
m10b:   mov         #$EF,OP1+3     ;yes, using the AD9851 so load these constants instead
        clr         OP1+2
m10a:   clr         OP1+1
        clr         OP1+0
        mov         BCD_Dat+6,OP2
        jmp         m_go

m100:   mov         #$C7,OP1+3
        mov         #$10,OP1+2
        lda         DDSchip            ;which DDS chip?
        beq         m100a              ;yes, using the AD9850 so just continue
        cmp         #1                 ;AD9851 w/ 6x PLL?
        beq         m100b
        mov         #$ED,OP1+3     ;no, using the AD9851 at 1x refclk so load these constants instead
        mov         #$37,OP1+2
        bra         m100a
m100b:  mov         #$52,OP1+3     ;Yes, using the AD9851 so load these constants instead
        mov         #$09,OP1+2
m100a:  clr         OP1+1
        clr         OP1+0
        mov         BCD_Dat+5,OP2
        jmp         m_go

m1K:    mov         #$C6,OP1+3
        mov         #$A7,OP1+2
        lda         DDSchip            ;which DDS chip?
        beq         m1Ka               ;yes, using the AD9850 so just continue
        cmp         #1                 ;AD9851 w/ 6x PLL?
        beq         m1Kb
        mov         #$3E,OP1+3     ;no, using the AD9851 at 1x refclk so load these constants instead
        mov         #$2F,OP1+2
        bra         m1Ka
m1Kb:   mov         #$35,OP1+3     ;yes, using the AD9851 so load these constants instead
        mov         #$5D,OP1+2
m1Ka:   clr         OP1+1
        clr         OP1+0
        mov         BCD_Dat+4,OP2
        jmp         m_go

m10K:   mov         #$B9,OP1+3
        mov         #$8D,OP1+2
        mov         #$06,OP1+1
        lda         DDSchip            ;which DDS chip?
        beq         m10Ka              ;yes, using the AD9850 so just continue
        cmp         #1                 ;AD9851 w/ 6x PLL?
        beq         m10Kb
        mov         #$68,OP1+3     ;no, using the AD9851 at 1x refclk so load these constants instead
        mov         #$D8,OP1+2
        mov         #$15,OP1+1
        bra         m10Ka
m10Kb:  mov         #$11,OP1+3     ;yes, using the AD9851 so load these constants instead
        mov         #$A4,OP1+2
        mov         #$03,OP1+1
m10Ka:  clr         OP1+0
        mov         BCD_Dat+3,OP2
        jmp         m_go

m100K:  mov         #$37,OP1+3
        mov         #$89,OP1+2
        mov         #$41,OP1+1
        lda         DDSchip            ;which DDS chip?
        beq         m100Ka             ;yes, using the AD9850 so just continue
        cmp         #1                 ;AD9851 w/ 6x PLL?
        beq         m100Kb
        mov         #$0E,OP1+3     ;no, using the AD9851 at 1x refclk so load these constants instead
        mov         #$74,OP1+2
        mov         #$DA,OP1+1
        bra         m100Ka
m100Kb: mov         #$AD,OP1+3     ;yes, using the AD9851 w/ 6x PLL so load these constants instead
        mov         #$68,OP1+2
        mov         #$24,OP1+1
m100Ka: clr         OP1+0
        mov         BCD_Dat+2,OP2
        bra         m_go

m1M:    mov         #$29,OP1+3
        mov         #$5C,OP1+2
        mov         #$8F,OP1+1
        mov         #$02,OP1+0
        lda         DDSchip            ;which DDS chip?
        beq         m1Ma               ;yes, using the AD9850 so just continue
        cmp         #1                 ;AD9851 w/ 6x PLL?
        beq         m1Mb
        mov         #$89,OP1+3     ;no using the AD9851 at 1x refclk so load these constants instead
        mov         #$88,OP1+2
        mov         #$88,OP1+1
        mov         #$08,OP1+0
        bra         m1Ma
M1Mb:   mov         #$C1,OP1+3     ;yes, using the AD9851 w/6x PLL so load these constants instead
        mov         #$16,OP1+2
        mov         #$6C,OP1+1
        mov         #$01,OP1+0
m1Ma:   mov         BCD_Dat+1,OP2
        bra         m_go

m10M:   mov         #$9A,OP1+3
        mov         #$99,OP1+2
        mov         #$99,OP1+1
        mov         #$19,OP1+0
        lda         DDSchip            ;which DDS chip?
        beq         m10Ma              ;yes, using the AD9850 so just continue
        cmp         #1                 ;AD9851 w/ 6x PLL?
        beq         m10Mb
        mov         #$55,OP1+3     ;no, using the AD9851 at 1x refclk so load those constants instead
        mov         #$55,OP1+2
        mov         #$55,OP1+1
        mov         #$55,OP1+0
        bra         m10Ma
m10Mb:  mov         #$8E,OP1+3     ;yes, using the AD9851 so load those constants instead
        mov         #$E3,OP1+2
        mov         #$38,OP1+1
        mov         #$0E,OP1+0
m10Ma:  mov         BCD_Dat+0,OP2
        bra         m_go

m_go:   mov         #4,temp2

digit_loop:
        clc
        ror         OP2
        bcc         noCarry
        jsr         Add32

noCarry:
        clc
        rol         OP1+3
        rol         OP1+2
        rol         OP1+1
        rol         OP1+0
        dbnz        temp2,digit_loop
        inc         temp1
        lda         temp1
        cmp         #1
        bne         _nc1
        jmp         m100
_nc1:   cmp         #2
        bne         _nc2
        jmp         m1K
_nc2:   cmp         #3
        bne         _nc3
        jmp         m10K
_nc3:   cmp         #4
        bne         _nc4
        jmp         m100K
_nc4:   cmp         #5
        bne         _nc5
        jmp         m1M
_nc5:   cmp         #6
        bne         _nc6
        jmp         m10M
_nc6:   rts

;***************************************************************************
; ADD32 -- 32 bit add
;
; The value in registers OP1(0:3) is added to the current 32 bit value
; in DDS_w1 [MSB] thru DDS_w4 [LSB]

Add32:
        lda     OP1+3
        add     DDS_w4
        sta     DDS_w4

        lda     OP1+2
        adc     DDS_w3
        sta     DDS_w3

        lda     OP1+1
        adc     DDS_w2
        sta     DDS_w2

        lda     OP1+0
        adc     DDS_w1
        sta     DDS_w1

        rts

;**************************************************************************
;***************      HCmon_LCD_disp        *******************************
;*     Routine called from HCmon to display message to LCD                *
;**************************************************************************

HCmon_LCD_disp:
        rts

mon_msg1:
        fcb     'HCmonitor v3.0',0

mon_msg2:
        fcb     '   N2APB',0



$page
;**************************************************************************
;*****************       DELAY ROUTINES     *******************************
;**************************************************************************

;--------------------------------------------------------------------------
;   Delay -- This subroutine performs a simple software delay loop based upon
;            the value passed in ACC. The following timing calculation applies:
;               delay = ((ACC * 74) + 12) (tcyc)

delay:
        psha                           ;[2] save delay parameter temporarily
delay1:
        lda #22T                       ;[2] initialize 5us loop counter
                                       ;     (repeat for timing)
delay2:
        dbnza delay2                   ;[3] decrement inner delay loop counter
        dbnz 1,sp,delay1               ;[6] decrement outer delay loop counter
        pula                           ;[2] deallocate local variable
        rts                            ;[4] return

;--------------------------------------------------------------------------
; Wait500ms -- Waits here for (10 x 50 x 1ms) = 500ms

wait500ms:
        lda     #10
        sta     count_hi
w500m1: lda     #$50
        sta     count_lo
w500m2: jsr     wait1ms
        dbnz    count_lo,w500m2
        dbnz    count_hi,w500m1
        rts

;--------------------------------------------------------------------------
; Wait100ms-- Waits here for 100ms

wait100ms:
        lda     #100
w100a:  psha
        bsr     wait1ms
        pula
        dbnza   w100a
        rts

;--------------------------------------------------------------------------
; Wait64ms-- Waits here for 64ms

wait64ms:
        lda     #64
w64a:   psha
        bsr     wait1ms
        pula
        dbnza   w64a
        rts

;--------------------------------------------------------------------------
; Wait50ms-- Waits here for 50ms

wait50ms:
        mov     #16,timeout1
w50a:   tst     timeout1
        bne     w50a
        rts

;--------------------------------------------------------------------------
; Wait32ms-- Waits here for 32ms

wait32ms:
        lda     #32
w32a:   psha
        bsr     wait1ms
        pula
        dbnza   w32a
        rts

;--------------------------------------------------------------------------
; Wait64ms-- Waits here for 16ms

wait16ms:
        lda     #16
w16a:   psha
        bsr     wait1ms
        pula
        dbnza   w16a
        rts

;--------------------------------------------------------------------------
; Wait10ms-- Waits here for 10ms

Wait10ms:
        lda     #10
w10a:   psha
        bsr     wait1ms
        pula
        dbnza   w10a
        rts

;--------------------------------------------------------------------------
; Wait1ms-- Waits here for 1ms

wait1ms:
        lda     #10
w1msa:  psha
        bsr     wait100us
        pula
        dbnza   w1msa
        rts

;--------------------------------------------------------------------------
; Wait100us -- Waits here for 100us
;              1 cycle = .11454545 US
;              (137 x (5 x .11454545)) + (6 x .11454545)
;              (174 x .5727) + .687 = 100 us

wait100us:
        lda     #174       ; [2]
w100u:  deca               ; [1]
        nop                ; [1]
        bne     w100u      ; [3]
        rts                ; [4]

;--------------------------------------------------------------------------
; Wait1us -- Waits here for 1us

wait1us:
        lda     us1
        dbnza   *
        rts

;*************************************************************
;*****************     TIMER MODULE    ***********************
;*************************************************************

;-------------------------------------------------------------
; Init_Timer - Turns on timer A channel 1 for an Output
;              Compare in 1ms. Creates an interrupt
;              vector to the Timer ISR (T_ISR_A1), where Timeout1,
;              Timeout2 and Timeout3 are decremeted. If a timer
;              reaches 0, then a user-specified operation is
;              done.  Thus, the user can either set a timer in his
;              code and sit there waiting on it to become 0, or
;              he may place the timeout action routine right here
;              in the Timer Handlers.  We use the latter technique
;              for the "heartbeat" LED blink. We set Timeout1 to
;              255 in the Timer1 Handler below, and when 255 ticks
;              of the 1ms timer interrupt occur, we toggle the state
;              of the heartbeat LED and reset Timeout1 to occur again
;              in another 255 ticks.

Init_Timer:
       mov     #$36,tasc           ; Timer A cleared + stopped. Clicks every 64 bus cycles
       mov     #0,tach1H           ; Set Output Compare to happen in 136 clicks (1ms)
       mov     #136,tach1L         ; after we start the timer.
       mov     #$54,tasc1          ; Set Timer A Chan 1 for Output Compare operation.
       mov     #$06,tasc           ; Start the timer
       rts

;*********************************************************************
; T_ISR_A1 - 1ms Timer Interrupt Service Routine for Timer A Chan 1
;*********************************************************************

T_ISR_A1:
       pshh
       pshx
       psha

       bclr    7,tasc1              ;clear OC flag
       ldhx    tach1h
       aix     #127                 ;setup next interrupt in (127+9) clicks (1ms)
       aix     #9
       sthx    tach1h

Check_t1:
       tst     timeout1
       beq     check_t2             ;Timeout 1 currently active?
       dec     timeout1             ;yes
       bne     check_t2             ;finished counting down?
       jsr     t1_handler           ;yes - Execute Timeout 1 handler

Check_t2:
       tst     timeout2
       beq     check_t3             ;Timeout 2 currently active?
       dec     timeout2             ;yes
       bne     check_t3             ;finished counting down?
       jsr     t2_handler           ;yes - Execute Timeout 2 handler

Check_t3:
       tst     timeout3
       beq     done_A1              ;Timeout 3 currently active?
       dec     timeout3             ;yes
       bne     done_A1              ;finished counting down?
       jsr     t3_handler           ;yes - Execute Timeout 3 handler

done_A1:
       pula
       pulx
       pulh
       rti

;---------------------------------------------------------------
; Timer A Chan 1 (TACH1) Timeout Handlers
;                    All the user has to do is set one of the
;                    timeout variables to desired delay n (1-255)
;                    and the timeout handler will be executed
;                    in n milliseconds. Setting the timeout
;                    variable from within the handler will
;                    cause a periodic timeout as shown in
;                    timeout 1.  (This is used to driver the
;                    Heartbeat LED.)

t1_handler:
       lda     #255
       sta     timeout1
       brclr   6,portF,t1a          ;toggle heartbeat LED
       bclr    6,portF
       rts
t1a:   bset    6,portF
       rts

t2_handler:                         ;used for the Piezo beep duration
       nop
       rts

t3_handler:
       nop
       rts

;*****************************************************************
; T_ISR_A0 - Timer Interrupt Service Routine for TACH0,
;            which is used to create the Piezo tone.
;            The technique is to:
;              a) turn on this timer (TACH0) with desired
;                 frequency (click count); and
;              b) set the timeout2 variable for beep duration (1-255 ms)
;*****************************************************************

T_ISR_A0:
       pshh
       pshx
       psha

       tst     timeout2             ;is timer2 currently active?
       bne     t0a                  ;yes, continue processing
       tst     timeout2+1           ;high byte was zero ... how about low byte?
       bne     t0a                  ;yes, go decrement it and restart the TACH0 timer

            ;turn off interrupt, turn off Keyline (or Sonalert) and exit
       bclr    6,tasc0              ;clear CH0IE (disable TACH0 interrupt)
       bra     done_A0

            ;timeout2 is active. Decrement it.
t0a:   ldhx    timeout2
       aix     #-1
       sthx    timeout2

            ;restart the Piezo tone timer (TACH0)
       bclr    7,tasc0              ;clear O.C. Flag
       ldhx    tach0h               ;restart the Piezo tone
       aix     #44                  ;setup another interrupt in 44 clicks (Piezo ~2.6 KHz)
       sthx    tach0h

done_A0:
       pula
       pulx
       pulh
       rti

**************************************************************
* DUMMY_ISR - Dummy Interrupt Service Routine.               *
*             Just does a return from interrupt.             *
**************************************************************
dummy_isr:
       nop
       rti           ; return

**************************************************************
* UTILS - Utilities and routines shared from HCmon           *
**************************************************************

      include "drivers.asm"     ;drivers for EEPROM, DDS

**************************************************************
* UTILS - Utilities and routines shared from HCmon           *
*         Just link to routines in HCmon                     *
*         (But remember to adjust jump addresses if HCmon    *
*         ever changes!)                                     *
**************************************************************

;        include "utils.asm"    ;this is the alternative (but replicates code!)

_sp_gets:     jmp   $F4DB
_gets:        jmp   $F4EE
_ahex2b:      jmp   $F53F
_a2nibhex:    jmp   $F54D
_dblchar:     jmp   $F556
_nib_to_asc:  jmp   $F561
_comma_sp:    jmp   $F56C
_pr_letter_x: jmp   $F577
_pr_pound:    jmp   $F57C
_pr_comma:    jmp   $F581
_crlf:        jmp   $F586
_byte_sp:     jmp   $F58F
_space:       jmp   $F592
_pr_nibble:   jmp   $F599
_putch:       jmp   $F59C
_pwordx:      jmp   $F5A5
_pword:       jmp   $F5AD
_pbyte_sp:    jmp   $F5B9
_pbytex:      jmp   $F5BD
_pbyte:       jmp   $F5BE
_ischar:      jmp   $F5CD
_getchne:     jmp   $F5D7
_getch:       jmp   $F5E1
_do_again:    jmp   $F5F6
_puts_lp:     jmp   $F602
_puts:        jmp   $F60A
_gethexbyte:  jmp   $F60E
_get_atbl:    jmp   $F62E
_tohex:       jmp   $F641
_ishex:       jmp   $F64A
_clr_spaces:  jmp   $F660
_next_hex:    jmp   $F66C
_first_hex:   jmp   $F677
_gethex:      jmp   $F67E
_readhex:     jmp   $F686
_numck_lp:    jmp   $F689
_dis_asm:     jmp   $F6FA




